<?php

function kixxl_wc_get_shop($config)
{
    $shop = preg_replace('#^https?://#', '', get_site_url());
    if ($config['APP_ENV'] !== 'local') {
        $shop = str_replace('/', '-', $shop);
    }
    return $shop;
}

/**
 * Sets custom HTTP headers for security.
 *
 * This function sets the Content Security Policy header to restrict
 * which domains can embed the page in an iframe, enhancing security
 * by preventing attacks.
 *
 * @return void
 */
function kixxl_wc_set_custom_headers()
{
    header("Content-Security-Policy: frame-ancestors https://localhost/wordpress/");
}

/**
 * Starts a PHP session if not already started.
 *
 * This function initializes a new session for the current user if one
 * does not already exist, allowing for session-based storage of user
 * data throughout the plugin's lifecycle.
 *
 * @return void
 */
function kixxl_start_session() {
    if ( ! session_id() && ! headers_sent() ) {
        session_start();
    }
}

/**
 * Creates an order and sends it to a Shopify webhook endpoint.
 *
 * This function retrieves order details from WooCommerce, shapes the data
 * according to Shopify requirements, and sends a POST request to a specified
 * Shopify webhook endpoint with the order data.
 *
 * @param int $orderId The WooCommerce order ID.
 */
function kixxl_wc_order_create(int $orderId, $order)
{
    try {
        global $shop, $config;
        // Get order details
        $orderUniqueId = kixxl_wc_generate_resource_id('order', $orderId);
        $orderData = $order->get_data();
        $order_items = $order->get_items();
        $custom_tags = '';
        error_log('$orderData.....');
        error_log(json_encode($orderData));


        $order->update_meta_data('_billing_phone_number', $orderData['billing']['phone']);
        $order->save();

        // Shape the data according to Shopify
        $items_data = [];

        foreach ($order_items as $item) {
            $product = $item->get_product(); // Get the product object
            $sku = $product ? $product->get_sku() : ''; // Get the SKU if product exists
            $subtotal = $order->get_line_subtotal($item, false, false);
            $quantity = $item->get_quantity();
            $unit_price = $subtotal / $quantity;

            if (!empty($item->get_meta('tags'))) {
                $custom_tags = (!empty($custom_tags)) ? ($custom_tags . ',' . $item->get_meta('tags')) : ($custom_tags . $item->get_meta('tags'));
            }

            $items_data[] = [
                'id' => $item->get_id(),
                'product_id' => kixxl_wc_generate_resource_id('product', $item->get_product_id()),
                'name' => $item->get_name(),
                'quantity' => $quantity,
                'variant_id' => kixxl_wc_generate_resource_id('variant', $item->get_variation_id()),
                'price' => $unit_price,
                'sku' => $sku,
                'properties' => [
                    [
                        "name" => "_app_name",
                        "value" => $item->get_meta('_app_name'),
                    ],
                    [
                        "name" => "sheet_preview",
                        "value" => $item->get_meta('sheet_preview'),
                    ],
                    [
                        "name" => "edit_sheet",
                        "value" => $item->get_meta('edit_sheet'),
                    ],
                    [
                        "name" => "sheet_type",
                        "value" => $item->get_meta('sheet_type'),
                    ],
                    [
                        "name" => "_actual_gang_sheet",
                        "value" => $item->get_meta('_actual_gang_sheet'),
                    ],
                ],
            ];
        }

        $shipping = $order->get_address('shipping');
        $shipping_lines = $order->get_shipping_methods();
        $shipping_lines_data = [];
        foreach ($shipping_lines as $shipping_line) {
			$method_id = $shipping_line->get_method_id();
			if($method_id == 'pickup_location') {
				$shipping = null;
			}
			
			 // Get base data
			$data = $shipping_line->get_data();
			$data['title'] = $shipping_line->get_name();
            $data['method_id'] = $method_id;
			$shipping_lines_data[] = $data;
        }
        // Get order data
        $shopify_data = [
            'id' => $orderUniqueId,
            'order_number' => $orderId,
            'financial_status' => $orderData['status'],
            'current_total_price' => $orderData['total'],
            'price_without_tax' => $orderData['total'] - $orderData['total_tax'],
            'note' => $orderData['customer_note'],
            'created_at' => $orderData['date_created']->date('Y-m-d H:i:s'),
            'tags' => $custom_tags,
            "properties" => [],
            "customer" => [
                "id" => kixxl_wc_generate_resource_id('customer',$orderData['customer_id']),
                "legacyResourceId" => $orderData['customer_id'],
                "email" => $orderData['billing']['email'],
                "first_name" => $orderData['billing']['first_name'],
                "last_name" => $orderData['billing']['last_name'],
                "state" => "enabled",
                "note" => null,
                "verified_email" => true,
                "multipass_identifier" => null,
                "tax_exempt" => false,
                "phone" => $orderData['billing']['phone'],
                "currency" => $orderData['currency'],
                "default_address" => [
                    "id" => uniqid(),
                    "customer_id" => kixxl_wc_generate_resource_id('customer',$orderData['customer_id']),
                    "first_name" => $orderData['billing']['first_name'],
                    "last_name" => $orderData['billing']['last_name'],
                    "address1" => $orderData['billing']['address_1'],
                    "address2" => $orderData['billing']['address_2'],
                    "city" => $orderData['billing']['city'],
                    "province" => $orderData['billing']['state'],
                    "country" => $orderData['billing']['country'],
                    "zip" => $orderData['billing']['postcode'],
                    "phone" => $orderData['billing']['phone'],
                    "country_code" => $orderData['billing']['country'],
                    "default" => true,
                ],
            ],
            "shipping_address" => $shipping,
            "shipping_lines" => $shipping_lines_data,
            "line_items" => $items_data,
        ];

        error_log('order data sent to webhook...........');
        error_log(json_encode($shopify_data));
        // Endpoint URL
        $endpoint_url = $config["APP_URL"] . '/webhook/orders-create?platform=woocommerce&store='.$shop;
		
		$shopId = kixxl_wc_get_user_id_by_shop_name($shop);
        $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
        // Generate a JWT token
        $payload = [
            "user_id" => $shopId,
            "username" => $shop,
            "exp" => time() + 150, // Token expiration time (2.5 min)
        ];
        $token = kixxl_wc_create_jwt_token($payload, $shopPassw);

        $body = $shopify_data;
        $args = [
            'method' => 'POST',
            'sslverify' => false,
            'body' => json_encode($body),
            'headers' => [
                'x-woocommerce-shop-domain' => "$shop", //x-woo-shop-domain
				'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
        ];

        // Make a POST request to the endpoint
        $response = wp_remote_post($endpoint_url, $args);
        error_log('orders-placed shopify data');
        error_log(json_encode($shopify_data));
        error_log('order placed request ...........');
        error_log(json_encode($response));

        // Check if the request was successful
        if (!is_wp_error($response)) {
            // Request was successful, handle the response if needed
            $response_body = wp_remote_retrieve_body($response);
            // Handle response data
        } else {
            // Request failed, handle the error
            $error_message = $response->get_error_message();
            var_dump($error_message);
        }
    } catch (Exception $e) {
        error_log('Order create webhook called...');
        error_log($e->getMessage());
    }
}

/**
 * Handles the order status change event.
 *
 * This function is triggered when the status of a WooCommerce order changes.
 * It retrieves order details, formats the data according to Shopify data, and sends a
 * POST request to update the order status on a connected store.
 *
 * @param int $orderId The ID of the WooCommerce order.
 * @param string $old_status The previous status of the order.
 * @param string $new_status The new status of the order.
 *
 * @return void
 */
function kixxl_wc_order_status_changed(int $orderId, string $old_status, string $new_status)
{
    try {
        global $shop, $config;
        $new_status = str_replace('-', ' ', $new_status);
        // Get order details
        $order = wc_get_order($orderId);

        $orderUniqueId = kixxl_wc_generate_resource_id('order', $orderId);
        $orderData = $order->get_data();
        $order_items = $order->get_items();

        // Shape the data according to Shopify
        $items_data = [];
        $custom_tags = '';

        foreach ($order_items as $item) {
            $product = $item->get_product(); // Get the product object
            $sku = $product ? $product->get_sku() : ''; // Get the SKU if product exists
            $subtotal = $order->get_line_subtotal($item, false, false);
            $quantity = $item->get_quantity();
            $unit_price = $subtotal / $quantity;

            if (!empty($item->get_meta('tags'))) {
                $custom_tags = $item->get_meta('tags');
            }

            $items_data[] = [
                'id' => $item->get_id(),
                'product_id' => kixxl_wc_generate_resource_id('product', $item->get_product_id()),
                'name' => $item->get_name(),
                'quantity' => $quantity,
                'variant_id' => kixxl_wc_generate_resource_id('variant', $item->get_variation_id()),
                'price' => $unit_price,
                'sku' => $sku,
                'properties' => [
                    [
                        "name" => "_app_name",
                        "value" => $item->get_meta('_app_name'),
                    ],
                    [
                        "name" => "sheet_preview",
                        "value" => $item->get_meta('sheet_preview'),
                    ],
                    [
                        "name" => "edit_sheet",
                        "value" => $item->get_meta('edit_sheet'),
                    ],
                ],
            ];
        }

        $shipping = $order->get_address('shipping');
        $shipping_lines = $order->get_shipping_methods();
        $shipping_lines_data = [];
        foreach ($shipping_lines as $shipping_line) {
			$method_id = $shipping_line->get_method_id();
			if($method_id == 'pickup_location') {
				$shipping = null;
			}
			
			 // Get base data
			$data = $shipping_line->get_data();
			$data['title'] = $shipping_line->get_name();
            $data['method_id'] = $method_id;
			$shipping_lines_data[] = $data;
        }
        // Get order data
        $shopify_data = [
            'id' => $orderUniqueId,
            'order_number' => $orderId,
            'financial_status' => $new_status,
            'current_total_price' => $orderData['total'],
            'price_without_tax' => $orderData['total'] - $orderData['total_tax'],
            'note' => $orderData['customer_note'],
            'created_at' => $orderData['date_created']->date('Y-m-d H:i:s'),
            'tags' => $custom_tags,
            "properties" => [],
            "customer" => [
                "id" => kixxl_wc_generate_resource_id('customer',$orderData['customer_id']),
                "email" => $orderData['billing']['email'],
                "first_name" => $orderData['billing']['first_name'],
                "last_name" => $orderData['billing']['last_name'],
                "state" => "enabled",
                "note" => null,
                "verified_email" => true,
                "multipass_identifier" => null,
                "tax_exempt" => false,
                "phone" => $orderData['billing']['phone'],
                "currency" => $orderData['currency'],
                "default_address" => [
                    "id" => uniqid(),
                    "customer_id" => kixxl_wc_generate_resource_id('customer',$orderData['customer_id']),
                    "first_name" => $orderData['billing']['first_name'],
                    "last_name" => $orderData['billing']['last_name'],
                    "address1" => $orderData['billing']['address_1'],
                    "address2" => $orderData['billing']['address_2'],
                    "city" => $orderData['billing']['city'],
                    "province" => $orderData['billing']['state'],
                    "country" => $orderData['billing']['country'],
                    "zip" => $orderData['billing']['postcode'],
                    "phone" => $orderData['billing']['phone'],
                    "country_code" => $orderData['billing']['country'],
                    "default" => true,
                ],
            ],
            "shipping_address" => $shipping,
            "shipping_lines" => $shipping_lines_data,
            "line_items" => $items_data,
        ];

        // Endpoint URL
        $endpoint_url = $config["APP_URL"] . '/webhook/orders-updated?platform=woocommerce&store='.$shop;
		
		$shopId = kixxl_wc_get_user_id_by_shop_name($shop);
        $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
        // Generate a JWT token
        $payload = [
            "user_id" => $shopId,
            "username" => $shop,
            "exp" => time() + 150, // Token expiration time (2.5 min)
        ];
        $token = kixxl_wc_create_jwt_token($payload, $shopPassw);

        $body = $shopify_data;
        $args = [
            'method' => 'POST',
            'sslverify' => false,
            'body' => json_encode($body),
            'headers' => [
                'x-woocommerce-shop-domain' => "$shop", //x-woo-shop-domain
				'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
        ];

        // Make a POST request to the endpoint
        $response = wp_remote_post($endpoint_url, $args);
        error_log('order updated request ...........');
        error_log(json_encode($response));

        // Check if the request was successful
        if (!is_wp_error($response)) {
            // Request was successful, handle the response if needed
            $response_body = wp_remote_retrieve_body($response);
            // Handle response data
        } else {
            // Request failed, handle the error
            $error_message = $response->get_error_message();
            var_dump($error_message);
        }
    } catch (Exception $e) {
        error_log('Order update webhook called...');
        error_log($e->getMessage());
    }

}

/**
 * Handles the product update event.
 *
 * This function is triggered when a WooCommerce product is updated.
 * It retrieves product details, formats the data for Shopify, and sends
 * a POST request to update the product information on a connected store.
 *
 * @param int $productId The ID of the updated product.
 *
 * @return void|WP_Error Returns a WP_Error if the product is not found.
 */
function kixxl_wc_product_updated(int $productId)
{
    try {
        global $config, $shop;

        $product = wc_get_product($productId);
        $dbProductId = kixxl_wc_generate_resource_id('product', $productId);

        if (!$product) {
            return new WP_Error('product_not_found', __('Product not found.'), ['status' => 404]);
        }

        $product_data = new stdClass();
        $product_data->id = $dbProductId;
        $product_data->title = $product->get_name();
        $product_data->status = 'ACTIVE';

        $image_id = $product->get_image_id();
        $image_url = wp_get_attachment_image_url($image_id, 'full');
        $product_data->image = new stdClass();
        $product_data->image->src = $image_url;

        $product_data->productType = "";

        $variants_data = [];
        if ($product->is_type('variable')) {
            $variations = $product->get_available_variations();
            foreach ($variations as $key => $variation) {
                $variationTitleConcat = "";
                $selectedOptions = [];
                foreach ($variation['attributes'] as $key => $value) {
                    $attribute = substr($key, strlen("attribute_"));
                    if (strpos($attribute, "size")) {
                        $attribute = "Size";
                    }
                    $selectedOptions[] = [
                        "name" => ucfirst($attribute),
                        "value" => $value,
                    ];
                    if (!empty($variationTitleConcat)) {
						$variationTitleConcat .= ' / ';
					}
					$variationTitleConcat .= $value;
                }
                $variantId = kixxl_wc_generate_resource_id('variant', $variation['variation_id']);

                $variants_data[] = (object)[
                    'id' => $variantId,
                    'title' => $variationTitleConcat,
                    'price' => $variation['display_price'],
                    'sku' => $variation['sku'],
                    'displayName' => $product_data->title . ' - ' . $variation['display_price'] . '/' . $variationTitleConcat,
                    'position' => $key == 0 ? 1 : $key,
                    'selectedOptions' => $selectedOptions,
                ];
            }
        } else {
            $variantId = kixxl_wc_generate_resource_id('variant', $productId);
            $variants_data[] = (object)[
                'id' => $variantId,
                'title' => $product->get_name(),
                'price' => $product->get_price(),
                'sku' => $product->get_sku(),
            ];
        }

        $product_data->variants = $variants_data;


        // Endpoint URL
        $endpoint_url = $config["APP_URL"] . '/webhook/products-update?platform=woocommerce&store='.$shop;
		
		$shopId = kixxl_wc_get_user_id_by_shop_name($shop);
        $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
        // Generate a JWT token
        $payload = [
            "user_id" => $shopId,
            "username" => $shop,
            "exp" => time() + 150, // Token expiration time (2.5 min)
        ];
        $token = kixxl_wc_create_jwt_token($payload, $shopPassw);
        $body = $product_data;
        $args = [
            'method' => 'POST',
            'sslverify' => false,
            'body' => json_encode($body),
            'headers' => [
                'x-woocommerce-shop-domain' => "$shop", //x-woo-shop-domain
				'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
        ];

        // Make a POST request to the endpoint
        $response = wp_remote_post($endpoint_url, $args);
        error_log('product updated request ...........');
        error_log(json_encode($response));

        // Check if the request was successful
        if (!is_wp_error($response)) {
            // Request was successful, handle the response if needed
            $response_body = wp_remote_retrieve_body($response);
            // Handle response data
        } else {
            // Request failed, handle the error
            $error_message = $response->get_error_message();
            var_dump($error_message);
        }
    } catch (Exception $e) {
        error_log('Product update webhook called...');
        error_log($e->getMessage());
    }

}

/**
 * Handles plugin uninstallation.
 *
 * This function is called when the plugin is uninstalled. It sends a
 * POST request to the Laravel application indicating that the app has
 * been uninstalled, and cleans up user session data.
 *
 * @return void
 */
function kixxl_wc_plugin_uninstall()
{
    try {
        if (session_status() == PHP_SESSION_NONE) {
            session_start();
        }
        $config = kixxl_wc_get_config();

        $shop = kixxl_wc_get_shop($config);

        // Endpoint URL
        $endpoint_url = $config["APP_URL"] . '/webhook/app-uninstalled?platform=woocommerce&store='.$shop;
		$shopId = kixxl_wc_get_user_id_by_shop_name($shop);
        $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
		
        // Generate a JWT token
        $payload = [
            "user_id" => $shopId,
            "username" => $shop,
            "exp" => time() + 150, // Token expiration time (2.5 min)
        ];
        $token = kixxl_wc_create_jwt_token($payload, $shopPassw);
        $body = new stdClass();
        $args = [
            'method' => 'POST',
            'sslverify' => false,
            'body' => json_encode($body),
            'headers' => [
                'x-woocommerce-shop-domain' => "$shop", //x-woo-shop-domain
				'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
        ];

        // Make a POST request to the endpoint
        $response = wp_remote_post($endpoint_url, $args);
        error_log('Uninstall webhook request ...........');
        error_log(json_encode($response));

        // Check if the request was successful
        if (!is_wp_error($response)) {
            // Request was successful, handle the response if needed
            $response_body = wp_remote_retrieve_body($response);
            // Handle response data
        } else {
            // Request failed, handle the error
            $error_message = $response->get_error_message();
            var_dump($error_message);
            if (isset($_SESSION['kixxl_wc_user_details'])) {
                unset($_SESSION['kixxl_wc_user_details']);
            }

        }
    } catch (Exception $e) {
        error_log('Uninstall webhook failed...');
        error_log($e->getMessage());
    }
}

/**
 * Retrieves the user's plan ID.
 *
 * This function fetches the user's plan ID by calling a specific method
 * that retrieves user details and sets the token in the options table.
 *
 * @return mixed The user's plan ID or null if not found.
 */
function get_user_plan_id()
{
    global $config;
    $shop = kixxl_wc_get_shop($config);

    $data = kixxl_wc_get_user_and_set_token_to_options_table_kixxl_app($shop);

    if ($data['errors']) {
        error_log('Plan Not Found.........');
    }

    return [
        'plan_id' => $data['plan_id'],
        'permissions' => $data['permissions'],
    ];
}

/**
 * Adds the Kixxl admin menu in the WordPress dashboard.
 *
 * This function creates a main menu and submenus for the Kixxl plugin
 * in the WordPress admin area. It checks the user's plan ID to conditionally
 * display certain submenus.
 *
 * @return void
 */
function kixxl_wc_kixxl_add_admin_menu()
{
    $user_plan = get_user_plan_id();
    $user_plan_id = $user_plan['plan_id'];
    $user_permissions = $user_plan['permissions'];

    add_menu_page(
        'Kixxl', // Page title
        'Kixxl', // Menu title
        'manage_options', // Capability
        'kixxl', // Menu slug
        'kixxl_wc_kixxl_render_admin_page', // Callback function
        'dashicons-admin-generic', // Icon URL (you can add a custom icon here)
        6 // Position
    );

    if (!empty($user_plan_id)) {
        add_submenu_page(
            'kixxl', // Parent slug
            'Kixxl Products', // Page title
            'Products', // Menu title
            'manage_options', // Capability
            'kixxl_products', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        add_submenu_page(
            'kixxl', // Parent slug
            'Kixxl Library', // Page title
            'Library', // Menu title
            'manage_options', // Capability
            'kixxl_library', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        add_submenu_page(
            'kixxl', // Parent slug
            'Fonts', // Page title
            'Fonts', // Menu title
            'manage_options', // Capability
            'kixxl_fonts', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        add_submenu_page(
            'kixxl', // Parent slug
            'Kixxl Customers', // Page title
            'Customers', // Menu title
            'manage_options', // Capability
            'kixxl_customers', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        add_submenu_page(
            'kixxl', // Parent slug
            'Kixxl Orders', // Page title
            'Orders', // Menu title
            'manage_options', // Capability
            'kixxl_orders', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        add_submenu_page(
            'kixxl', // Parent slug
            'Kixxl Settings', // Page title
            'Settings', // Menu title
            'manage_options', // Capability
            'kixxl_settings', // Menu slug
            'kixxl_wc_kixxl_render_admin_page' // Callback function
        );
        if($user_permissions->can_access_admin_builder) {
            add_submenu_page(
                'kixxl', // Parent slug
                'Kixxl Singles Builder', // Page title
                'Singles Builder', // Menu title
                'manage_options', // Capability
                'kixxl_singles_builder', // Menu slug
                'kixxl_wc_kixxl_render_admin_page' // Callback function
            );
        }
    }
}

/**
 * Renders the admin page for the Kixxl plugin.
 *
 * This function generates the HTML for the Kixxl admin page, including
 * an iframe that loads the relevant app URL based on the current user
 * plan and the current admin screen. It checks for the presence of an
 * authentication token and fetches it if necessary.
 *
 * @return void
 */
function kixxl_wc_kixxl_render_admin_page()
{
    global $config;
    $current_screen = get_current_screen();
    $user_plan = get_user_plan_id();
    $user_plan_id = $user_plan['plan_id'];
    $user_permissions = $user_plan['permissions'];
    $url = "/app/home";
    if (!empty($user_plan_id)) {
        switch ($current_screen->base) {
            case 'kixxl_page_kixxl_library':
                $url = '/app/library';
                break;
            case 'kixxl_page_kixxl_products':
                $url = '/app/products';
                break;
            case 'kixxl_page_kixxl_orders':
                $url = '/app/orders';
                break;
            case 'kixxl_page_kixxl_fonts':
                $url = '/app/fonts';
                break;
            case 'kixxl_page_kixxl_settings':
                $url = '/app/settings';
                break;
            case 'kixxl_page_kixxl_customers':
                $url = '/app/customers';
                break;
            case 'kixxl_page_kixxl_singles_builder':
                if ($user_permissions->can_access_admin_builder) {
                    $url = '/app/admin-builder';
                }
                break;
            default:
                $url = '/app/home';
                break;
        }
    }

    $shop = kixxl_wc_get_shop($config);

    $shopId = kixxl_wc_get_user_id_by_shop_name($shop);
    $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
    if (empty($shopId) || empty($shopPassw)) {
        // Store a flag to trigger the admin notice on the next page load
        add_option('kixxl_wc_activation_error', true);
        // Redirect to the plugins page
        wp_redirect(admin_url('plugins.php'));
        exit;
    }

    // Generate a JWT token
    $payload = [
        "user_id" => $shopId,
        "username" => $shop,
        "exp" => time() + 150, // Token expiration time (2.5 min)
    ];
    $token = kixxl_wc_create_jwt_token($payload, $shopPassw);

    if ($config['APP_ENV'] == 'local') {
        $url = $config["APP_URL"] . $url . "?store=$shop&token=$token&platform=woocommerce&ancestor=" . $config['ancestors'];
    } else {
        $currentSite = get_site_url();
        error_log('json_encode($currentSite)');
        error_log(json_encode($currentSite));
        $url = $config["APP_URL"] . $url . "?store=$shop&token=$token&ancestor=$currentSite&platform=woocommerce";
    }
    ?>

    <div class="wrap">
        <iframe src="<?php echo $url ?>" width="100%" style="border: none; height: 100vh"
                allow="clipboard-read; clipboard-write"></iframe>
    </div>
    <?php
}

/**
 * Creates JWT Token
 *
 * @param  $payload
 * @param  $secretKey
 *
 * @return string The decoded string.
 */
function kixxl_wc_create_jwt_token($payload, $secretKey)
{
    $base64UrlHeader = kixxl_wc_base64_url_encode(json_encode(["alg" => "HS256", "typ" => "JWT"]));
    $base64UrlPayload = kixxl_wc_base64_url_encode(json_encode($payload));
    $base64UrlSignature = hash_hmac('sha256', $base64UrlHeader . '.' . $base64UrlPayload, $secretKey, true);
    $base64UrlSignature = kixxl_wc_base64_url_encode($base64UrlSignature);

    return $base64UrlHeader . '.' . $base64UrlPayload . '.' . $base64UrlSignature;
}

/**
 * Encodes a string using URL-safe Base64 encoding.
 *
 * This function encodes the input string using Base64 and then replaces
 * characters that are not URL-safe
 * ('+' and '/') with URL-safe alternatives ('-' and '_'). It also removes any
 * trailing '=' characters.
 *
 * @param string $input The string to be encoded.
 *
 * @return string The URL-safe Base64 encoded string.
 */
function kixxl_wc_base64_url_encode($input)
{
    return rtrim(strtr(base64_encode($input), '+/', '-_'), '=');
}

/**
 * Validates JWT Token
 *
 * @param string $token
 * @param string $userPassw
 *
 * @return string The decoded string.
 */
function kixxl_wc_validate_token($token, $userPassw)
{
    // Implementation for validating JWT
    list($base64UrlHeader, $base64UrlPayload, $base64UrlSignature) = explode('.', $token);

    $signature = kixxl_wc_base64_url_decode($base64UrlSignature);
    $expectedSignature = hash_hmac('sha256', $base64UrlHeader . '.' . $base64UrlPayload, $userPassw, true);

    return hash_equals($signature, $expectedSignature);
}

/**
 * Decodes a URL-safe Base64 encoded string.
 *
 * This function decodes a URL-safe Base64 encoded string by replacing URL-safe
 * characters ('-' and '_') with their original Base64 characters ('+' and
 * '/'). It also adds any necessary padding with '=' characters.
 *
 * @param string $input The URL-safe Base64 encoded string to be decoded.
 *
 * @return string The decoded string.
 */
function kixxl_wc_base64_url_decode($data)
{
    $base64 = strtr($data, '-_', '+/');
    $base64Padded = str_pad($base64, strlen($base64) % 4, '=', STR_PAD_RIGHT);
    return base64_decode($base64Padded);
}

/**
 * Registers the current user with the Kixxl app.
 *
 * This function sends a request to the Kixxl API to register the
 * current user, capturing their name, email, and timezone. On success,
 * it stores the returned shop ID and token in the WordPress options
 * table.
 *
 * @return void
 */
function kixxl_wc_register_user_to_kixxl_app()
{
//    global $config;
    // Start the session if it's not already started
    if (session_status() == PHP_SESSION_NONE) {
        session_start();
    }
    $config = kixxl_wc_get_config();

    $current_user = wp_get_current_user();

    if ($current_user->ID === 0) {
        error_log("No user is currently logged in.");
        return;
    }
    $email = $current_user->data->user_email;
    $shopName = kixxl_wc_get_shop($config);
    $user_plan = get_user_plan_id();
    $user_plan_id = $user_plan['plan_id'];

    // Prepare the data for the API request
    $api_url = $config["APP_URL"] . '/api/register-user'; // You can save the URL in the options table
    $body = array(
        'name' => $shopName,
        'email' => $email,
        'plan_id' => $user_plan_id,
        'timezone' => date_default_timezone_get(),

    );

    $response = wp_remote_post($api_url, array(
        'method' => 'POST',
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body' => json_encode($body),
        'timeout' => 60,
        'sslverify' => false,
    ));

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        error_log("User registration failed: $error_message");
    } else {
        $response_body = wp_remote_retrieve_body($response);
        $response_body = json_decode($response_body);
        $user_details = $response_body->user;

        update_option('_kixxl_' . $user_details->name . '_shop_id', $user_details->id);

        // Generate a salt using a hash of the email and shop domain
        $salt = substr(sha1($email . $shopName), 0, 16);
        // Generate the password using crypt() with a salt
        $userPassword = 'wcat_' . crypt($email . $shopName, '$6$' . $salt); // $6$ indicates SHA-512 hash

        update_option('_kixxl_' . $user_details->name . '_shop_password', $userPassword);
        error_log("User registered successfully: " . json_encode($response_body));
    }
}

/**
 * Retrieves user details and sets the token in the options table.
 *
 * This function fetches the user information from the Kixxl API using
 * the current user's shop name. It updates the options table with the
 * user's token if the request is successful.
 *
 * @param string $shop The shop name associated with the current user.
 *
 * @return array An associative array containing errors and user plan ID.
 */
function kixxl_wc_get_user_and_set_token_to_options_table_kixxl_app(string $shop): array
{
    $config = kixxl_wc_get_config();

    $current_user = wp_get_current_user();

    if ($current_user->ID === 0) {
        error_log("No user is currently logged in.");
        return [];
    }
    // Prepare the data for the API request
    $api_url = $config["APP_URL"] . '/api/get-user'; // You can save the URL in the options table
    $body = array(
        'name' => $shop,
    );

    $response = wp_remote_post($api_url, array(
        'method' => 'POST',
        'headers' => array(
            'Content-Type' => 'application/json',
        ),
        'body' => json_encode($body),
        'timeout' => 60,
        'sslverify' => false,
    ));

    if (is_wp_error($response)) {
        $error_message = $response->get_error_message();
        error_log("User Fetch failed: $error_message");
    } else {
        $response_body = wp_remote_retrieve_body($response);
        $response_body = json_decode($response_body);

        update_option('_kixxl_' . $response_body->shop_name . '_token', $response_body->token);
        return [
            'errors' => false,
            'shopName' => $response_body->shop_name,
            'plan_id' => $response_body->plan_id,
            'permissions' => $response_body->permissions
        ];
    }

    return [
        'errors' => true,
        'shopName' => null,
        'plan_id' => null,
        'permissions' => false
    ];
}

/**
 * Generates unique id for kixxl
 *
 * This will generate unique number for resource(Product,Variant,Image,Order..etc)
 * combining Prefix (4 digit) + Resource Type ID (2 digit) + Shop ID (6 digit) + Resource ID (unlimited)
 *
 * @param  $resource_type
 * @param  $resource_id
 *
 * @return string
 */
function kixxl_wc_generate_resource_id($resource_type, $resource_id): string
{
    global $shop;

    // Define Resource Types with 2-digit IDs
    $resource_types = [
        'product' => '01',
        'order' => '02',
        'variant' => '03',
        'image' => '04',
        'option' => '05',
        'metafield' => '06',
        'draftOrder' => '07',
        'customer' => '08',
        'media' => '09',
        'guest' => '18'
    ];

    // Set the Prefix
    $prefix = '44';

    // Get the 2-digit Resource Type ID
    $resource_type_id = $resource_types[$resource_type];

    // Fetch shop ID and pad to 4 digits
    $shop_id = kixxl_wc_get_user_id_by_shop_name($shop);
    $shop_id = str_pad($shop_id, 4, '0', STR_PAD_LEFT);

    // No limit on the number of digits in $resource_id, but ensure it is a string
    $resource_id = (string)$resource_id;

    // Return the formatted Resource ID
    return $prefix . $resource_type_id . $shop_id . $resource_id;
}

/**
 * Decode resource id for WordPress
 *
 * This will extract the ResourceID by skipping the first 8 characters
 * (2 for prefix, 2 for ResourceType, 4 for StoreID)
 *
 * @param  $resource_id
 *
 * @return int
 */
function kixxl_wc_decode_resource_id($resource_id): int
{
    // Extract the ResourceID by skipping the first 8 characters (2 for prefix, 2 for ResourceType, 4 for StoreID)
    $resource_id = substr($resource_id, 8);

    // Convert to integer to remove any leading zeros
    return (int)$resource_id;
}

/**
 * Retrieves the user ID associated with a given shop name.
 *
 * This function fetches the shop ID stored in the WordPress options table
 * for the specified shop name.
 *
 * @param string $shop_name The name of the shop.
 *
 * @return mixed The shop ID or null if not found.
 */
function kixxl_wc_get_user_id_by_shop_name(string $shop_name)
{
    return get_option('_kixxl_' . $shop_name . '_shop_id');
}

//for Location and Publication function
function kixxl_wc_generate_unique_id(): string
{
    // Get the current time in microseconds
    $timeInMicroseconds = microtime(true);

    // Convert the time to an integer by removing the decimal point
    $integerTime = str_replace('.', '', $timeInMicroseconds);

    // Ensure the integerTime is at least 9 digits long by padding with zeros if needed
    $integerTime = str_pad($integerTime, 9, '0', STR_PAD_RIGHT);

    // Take the first 9 digits of the integerTime
    $integerTime = substr($integerTime, 0, 9);

    // Generate a 4-digit random number
    $randomNumber = rand(1000, 9999);

    // Concatenate the 9-digit time-based integer and the 4-digit random number
    return $integerTime . $randomNumber;
}

/**
 * Find a WooCommerce customer by email.
 *
 * This function attempts to locate a customer by:
 *  1. Checking the WordPress users table.
 *  2. Querying the WooCommerce customer lookup table.
 *
 * @param string $email The email address to search for.
 * @return WP_REST_Response Returns a REST response containing the customer data or an empty array if not found.
 */
 function find_customer_by_email($email)
    {
    global $wpdb;

    // Sanitize and validate email
    $email = trim(sanitize_email($email));
    if (empty($email)) {
        return rest_ensure_response([
                "body"   => ['customer' => []],
                "errors" => true,
        ]);
    }

    // Check WordPress users table
    $user = get_user_by('email', $email);
            if ($user) {
                $customer = new WC_Customer($user->ID);
            } else {
        // Check WooCommerce customer lookup table
                global $wpdb;
                    $customer = $wpdb->get_row($wpdb->prepare("
                        SELECT * FROM {$wpdb->prefix}wc_customer_lookup 
                        WHERE email = %s
                LIMIT 1
            ", $email)
        );
        }

        if (empty($customer)) {
            return rest_ensure_response([
                "body"   => ['customer' => []],
                "errors" => true,
            ]);
        }

        return rest_ensure_response($customer);
    }


function on_site_language_changed($old_value, $new_value) 
{
	try {
        global $shop, $config;

		// Gather site/shop details
		$shop_name       = get_option('blogname');
		$admin_email     = get_option('admin_email');
		$currency_code   = get_option('woocommerce_currency');
		$timezone = get_option('timezone_string');
		$timezone = !empty($timezone) ? $timezone : 'UTC';
		
		$full_locale = $new_value;
		$locale_parts = explode('_', $full_locale);
		$locale = isset($locale_parts[0]) && !empty($locale_parts[0]) ? $locale_parts[0] : 'en';

        // Get data
        $shopify_data = [
			'name' => $shop_name,
			'email' => $admin_email,
			'currency' => $currency_code,
			'iana_timezone' => $timezone,
			'primary_locale' => $locale,
        ];

        // Endpoint URL
        $endpoint_url = $config["APP_URL"] . '/webhook/shop-update?platform=woocommerce&store='.$shop;
		
		$shopId = kixxl_wc_get_user_id_by_shop_name($shop);
        $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
        // Generate a JWT token
        $payload = [
            "user_id" => $shopId,
            "username" => $shop,
            "exp" => time() + 150, // Token expiration time (2.5 min)
        ];
        $token = kixxl_wc_create_jwt_token($payload, $shopPassw);

        $body = $shopify_data;
        $args = [
            'method' => 'POST',
            'sslverify' => false,
            'body' => json_encode($body),
            'headers' => [
                'x-woocommerce-shop-domain' => "$shop", //x-woo-shop-domain
				'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json',
            ],
        ];

        // Make a POST request to the endpoint
        $response = wp_remote_post($endpoint_url, $args);

        // Check if the request was successful
        if (!is_wp_error($response)) {
            // Request was successful, handle the response if needed
            $response_body = wp_remote_retrieve_body($response);
            // Handle response data
        } else {
            // Request failed, handle the error
            $error_message = $response->get_error_message();
            var_dump($error_message);
        }
    } catch (Exception $e) {
        error_log('Shop Update webhook called...');
        error_log($e->getMessage());
    }
}

/**
 * These functions will handle all the proxy requests
 **/
// Global variables to maintain state
$kixxl_wc_base_url        = 'https://dev.gangifyapp.com';
$kixxl_wc_allowed_hosts   = [
    'dev.gangifyapp.com',
    'gangifyapp.com'
];
$kixxl_wc_default_headers = [
    'Accept: text/html,application/xhtml+xml,application/xml;q=0.9,image/avif,image/webp,*/*;q=0.8',
    'User-Agent: Gangify Proxy/1.0',
];
$kixxl_wc_request_data    = [];

/**
 * Initialize the proxy with configuration
 *
 * @param string $base_url Base URL to proxy requests to
 * @return void
 */
function kixxl_wc_init_proxy($base_url)
{
    global $kixxl_wc_base_url, $kixxl_wc_request_data;

    $kixxl_wc_base_url     = rtrim($base_url, '/');
    $kixxl_wc_request_data = kixxl_wc_capture_request_data();
}

/**
 * Capture request data
 *
 * @return array Request data including method, GET, POST, files, headers, and raw input
 */
function kixxl_wc_capture_request_data()
{
    return [
        'method'    => $_SERVER['REQUEST_METHOD'] ?? 'GET',
        'get'       => $_GET ?? [],
        'post'      => $_POST ?? [],
        'files'     => $_FILES ?? [],
        'headers'   => kixxl_wc_get_all_headers(),
        'raw_input' => file_get_contents('php://input')
    ];
}

/**
 * Get all request headers
 *
 * @return array All HTTP headers from the request
 */
function kixxl_wc_get_all_headers()
{
    $headers = [];
    //print_r($_SERVER);
    foreach ($_SERVER as $name => $value) {
        if (str_starts_with($name, 'HTTP_')) {
            $key           = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($name, 5)))));
            $headers[$key] = $value;
        }else if($name == 'CONTENT_TYPE'){
            $key           = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', $name))));
            $headers[$key] = $value;
        }

    }
    return $headers;
}

/**
 * Main proxy method - now handles all requests with cURL
 *
 * @param string|null $path Path to proxy to
 * @param array $query_params Optional query parameters to add to the request
 * @return void
 */
function kixxl_wc_proxy_request($path = null, $query_params = [])
{
    try {
        global $kixxl_wc_base_url, $kixxl_wc_request_data;

        // Sanitize and validate path
        $path = kixxl_wc_sanitize_path($path);

        // Construct full URL
        $fullUrl = $kixxl_wc_base_url . '/' . ltrim($path, '/');

        // Add query parameters if provided
        if (!empty($query_params)) {
            // Add ? or & depending on whether URL already contains parameters
            $separator = (strpos($fullUrl, '?') === false) ? '?' : '&';
            $fullUrl   .= $separator . http_build_query($query_params);
        }

        // Handle the request using cURL
        $response = kixxl_wc_curl_request($fullUrl);
        echo $response;

    } catch (Exception $e) {
        kixxl_wc_handle_error($e);
    }
}

/**
 * Process any request using cURL and return the response
 * Handles redirects internally through the proxy
 *
 * @param string $url URL to send the request to
 * @param int $redirectCount Counter to prevent infinite redirect loops
 * @return string Response from the remote server
 */
function kixxl_wc_curl_request($url, $redirectCount = 0)
{
    global $kixxl_wc_request_data, $kixxl_wc_default_headers, $kixxl_wc_allowed_hosts;

    // Prevent infinite redirect loops
    $maxRedirects = 10;
    if ($redirectCount > $maxRedirects) {
        throw new Exception("Too many redirects (maximum: $maxRedirects)");
    }

    // Initialize cURL session
    $ch = curl_init();

    // Set cURL options
    curl_setopt($ch, CURLOPT_URL, $url);

    // Set appropriate method
    $method = $kixxl_wc_request_data['method'];
    if ($redirectCount === 0) {
        switch ($method) {
            case 'POST':
                curl_setopt($ch, CURLOPT_POST, true);
                // Check for file uploads

                if (!empty($kixxl_wc_request_data['files'])) {
                    // Handle multipart/form-data with file uploads
                    $postFields = kixxl_wc_prepare_multipart_data(
                        $kixxl_wc_request_data['post'],
                        $kixxl_wc_request_data['files']
                    );
                    print_r($postFields);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
                    curl_setopt($ch, CURLOPT_SAFE_UPLOAD, true);

                } else {

                    $contentType = $kixxl_wc_request_data['headers']['Content-Type'] ?? '';
                    if (strpos($contentType, 'application/json') !== false) {
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $kixxl_wc_request_data['raw_input']);
                    }else if (strpos($contentType, 'multipart/form-data') !== false) {
                        curl_setopt($ch, CURLOPT_POSTFIELDS, $kixxl_wc_request_data['post']);
                    } else {
                        curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($kixxl_wc_request_data['post']));
                    }
                }
                break;

            case 'PUT':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PUT');
                curl_setopt($ch, CURLOPT_POSTFIELDS, $kixxl_wc_request_data['raw_input']);
                break;

            case 'DELETE':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'DELETE');
                break;

            case 'PATCH':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'PATCH');
                curl_setopt($ch, CURLOPT_POSTFIELDS, $kixxl_wc_request_data['raw_input']);
                break;

            case 'OPTIONS':
                curl_setopt($ch, CURLOPT_CUSTOMREQUEST, 'OPTIONS');
                break;

            case 'HEAD':
                curl_setopt($ch, CURLOPT_NOBODY, true);
                break;

            default: // GET
                curl_setopt($ch, CURLOPT_HTTPGET, true);
        }
    } else {
        // For redirects, always use GET
        curl_setopt($ch, CURLOPT_HTTPGET, true);
    }

    // Prepare headers to forward
    $headers = $kixxl_wc_default_headers;
    foreach ($kixxl_wc_request_data['headers'] as $name => $value) {
        // Skip some headers we don't want to proxy
        if (!in_array($name, ['Host', 'Connection'])) {
            $headers[] = "$name: $value";
        }
    }

    curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HEADER, true); // Capture response headers
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false); // Don't follow redirects automatically
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    curl_setopt($ch, CURLOPT_ENCODING , '');

    // Disable SSL verification - skip certificate validation
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
    curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
    curl_setopt($ch, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2); // Specify TLS version if needed


    // Execute cURL session and get the response
    $response = curl_exec($ch);

    // Get response info
    $info = curl_getinfo($ch);
    $httpCode = $info['http_code'];
    $headerSize = $info['header_size'];

    // Split headers and body
    $responseHeaders = substr($response, 0, $headerSize);
    $responseBody = substr($response, $headerSize);

    // Close cURL session
    curl_close($ch);

    // Handle redirects (status codes 301, 302, 303, 307, 308)
    if ($httpCode >= 300 && $httpCode < 400) {
        // Extract the Location header
        if (preg_match('/Location:(.*?)\r\n/i', $responseHeaders, $matches)) {
            $redirectUrl = trim($matches[1]);

            // Parse the redirect URL components
            $parsedUrl = parse_url($redirectUrl);

            // If the redirect URL is relative (doesn't have a scheme/host)
            if (empty($parsedUrl['scheme']) || empty($parsedUrl['host'])) {
                // Make it absolute based on the current URL
                $parsedCurrentUrl = parse_url($url);
                $baseUrl = $parsedCurrentUrl['scheme'] . '://' . $parsedCurrentUrl['host'];
                if (!empty($parsedCurrentUrl['port'])) {
                    $baseUrl .= ':' . $parsedCurrentUrl['port'];
                }

                // Handle different types of relative URLs
                if (strpos($redirectUrl, '/') === 0) {
                    // Absolute path
                    $redirectUrl = $baseUrl . $redirectUrl;
                } else {
                    // Relative path - get directory of current URL
                    $currentPath = isset($parsedCurrentUrl['path']) ? $parsedCurrentUrl['path'] : '/';
                    $currentDir = substr($currentPath, 0, strrpos($currentPath, '/') + 1);
                    $redirectUrl = $baseUrl . $currentDir . $redirectUrl;
                }
            }

            // Check if the redirect URL is allowed
            $parsedRedirect = parse_url($redirectUrl);
            $redirectHost = $parsedRedirect['host'] ?? '';

            if (in_array($redirectHost, $kixxl_wc_allowed_hosts)) {
                // Follow the redirect internally
                return kixxl_wc_curl_request($redirectUrl, $redirectCount + 1);
            } else {
                // Rewrite external redirects to go through our proxy
                global $kixxl_wc_base_url;
                $proxyPath = isset($parsedRedirect['path']) ? ltrim($parsedRedirect['path'], '/') : '';
                $proxyQuery = isset($parsedRedirect['query']) ? '?' . $parsedRedirect['query'] : '';

                // Generate the redirect URL through our proxy
                $baseUrl = isset($_SERVER['HTTP_HOST']) ? (isset($_SERVER['HTTPS']) ? 'https://' : 'http://') . $_SERVER['HTTP_HOST'] : '';
                $proxyUrl = rtrim($baseUrl, '/') . '/apps/gangify/' . $proxyPath . $proxyQuery;

                // Send redirect header
                header("Location: $proxyUrl", true, $httpCode);
                exit;
            }
        }
    }

    // For non-redirect responses, process headers
    $headersToSkip = ['transfer-encoding', 'connection', 'content-length', 'content-encoding'];

    // Parse and forward headers
    $headerLines = explode("\r\n", $responseHeaders);
    foreach ($headerLines as $headerLine) {
        if (empty($headerLine) || strpos($headerLine, ':') === false) {
            continue;
        }

        // Skip the HTTP status line and headers we don't want to forward
        if (strpos($headerLine, 'HTTP/') === 0) {
            continue;
        }

        // Extract header name
        list($headerName, $headerValue) = explode(':', $headerLine, 2);
        $headerName = trim($headerName);
        $headerValue = trim($headerValue);

        // Skip certain headers
        if (in_array(strtolower($headerName), $headersToSkip)) {
            continue;
        }

        // Skip Location headers (we handle them separately)
        if (strcasecmp($headerName, 'Location') === 0) {
            continue;
        }

        // Forward other headers
        header("$headerName: $headerValue");
    }

    // Set status code
    http_response_code($httpCode);

    return $responseBody;
}

/**
 * Prepare multipart form data including file uploads for cURL
 *
 * @param array $postData POST data
 * @param array $fileData FILES data
 * @return array Data formatted for cURL multipart upload
 */
function kixxl_wc_prepare_multipart_data($postData, $fileData)
{
    // When using CURLFile, curl will automatically set the correct multipart headers
    $result = $postData; // Start with all the POST data

    // Process file fields
    foreach ($fileData as $fieldName => $fileInfo) {
        // Handle multiple files for the same field
        if (is_array($fileInfo['name'])) {
            foreach ($fileInfo['name'] as $index => $name) {
                // Skip if no file was uploaded or there was an error
                if (empty($name) || $fileInfo['error'][$index] !== UPLOAD_ERR_OK) {
                    continue;
                }

                // For array file names like file[]
                if (substr($fieldName, -2) === '[]') {
                    $curlFieldName = $fieldName;
                } else {
                    $curlFieldName = "{$fieldName}[{$index}]";
                }

                $result[$curlFieldName] = new CURLFile(
                    $fileInfo['tmp_name'][$index],
                    $fileInfo['type'][$index],
                    $name
                );
            }
        } else {
            // Single file upload
            // Skip if no file was uploaded or there was an error
            if (empty($fileInfo['name']) || $fileInfo['error'] !== UPLOAD_ERR_OK) {
                continue;
            }

            $result[$fieldName] = new CURLFile(
                $fileInfo['tmp_name'],
                $fileInfo['type'],
                $fileInfo['name']
            );
        }
    }

    return $result;
}


/**
 * Sanitize the path to prevent directory traversal
 *
 * @param string|null $path Path to sanitize
 * @return string Sanitized path
 */
function kixxl_wc_sanitize_path($path)
{
    // Use current path if none provided
    if ($path === null) {
        $path = $_SERVER['REQUEST_URI'] ?? '';
        // Remove query string if present
        $path = preg_replace('/\?.*$/', '', $path);
    }

    // Prevent directory traversal
    $path = str_replace(['../', '..\\'], '', $path);

    return $path;
}

/**
 * Handle errors in the proxy
 *
 * @param Exception $e Exception to handle
 * @return void
 */
function kixxl_wc_handle_error($e)
{
    http_response_code(500);
    echo "<!DOCTYPE html><html><head><title>Proxy Error</title></head><body>";
    echo "<h1>Proxy Error</h1>";
    echo "<p>" . htmlspecialchars($e->getMessage()) . "</p>";
    echo "</body></html>";
    exit;
}