<?php

/**
 * General_Routes_Controller class.
 *
 * This class handles the registration of custom REST API routes for searching
 * customer information in a WooCommerce store. The routes are registered under
 * the namespace 'kwc/v1'.
 */
class General_Routes_Controller extends WP_REST_Controller
{

    /**
     * Registers the routes for the custom REST API endpoints.
     *
     * This function registers a route for searching customer information based
     * on the customer ID.
     */
    public function register_routes()
    {
        $version = '1';
        $namespace = 'kwc/v' . $version;

        register_rest_route($namespace, '/get-token', array(
            'methods' => 'GET',
            'callback' => [$this, 'kixxl_wc_get_token'],
            'permission_callback' => '__return_true'
        ));

        register_rest_route($namespace, '/fetch-shop-detail-from-api', array(
            'methods' => 'GET',
            'callback' => [$this, 'fetch_shop_detail_from_api'],
            'permission_callback' => '__return_true'
        ));

        register_rest_route($namespace, '/search-customer', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'search_customer'],
                'permission_callback' => '__return_true',
            ],
        ]);
		
		register_rest_route($namespace, '/get-shopify-customer', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_shopify_customer'],
                'permission_callback' => '__return_true',
            ],
        ]);
		
		register_rest_route($namespace, '/get-shopify-customers', [
            [
                'methods' => 'POST',
                'callback' => [$this, 'get_woo_customers'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/authenticate', array(
            'methods' => 'POST',
            'callback' => [$this, 'kixxl_authenticate'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/editor', array(
            'methods' => 'GET',
            'callback' => [$this, 'editor_load'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/login', array(
            'methods' => 'GET',
            'callback' => [$this, 'kixxl_wc_login_page'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/logout', array(
            'methods' => 'GET',
            'callback' => [$this, 'kixxl_wc_logout_page'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/kixxl_wc-wordpress-logout', array(
            'methods' => 'POST',
            'callback' => [$this, 'kixxl_wc_logout_customer'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/kixxl_wc-wordpress-login', array(
            'methods' => 'POST',
            'callback' => [$this, 'kixxl_wc_login_customer'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/register', array(
            'methods' => 'GET',
            'callback' => [$this, 'kixxl_wc_register_page'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/kixxl_wc-wordpress-register', array(
            'methods' => 'POST',
            'callback' => [$this, 'kixxl_wc_register_customer'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/kixxl_wc_temp_cart_store', [
            [
                'methods' => 'POST',
                'callback' => [$this, 'kixxl_store_temp_cart_data'],
                'permission_callback' => '__return_true'
            ],
        ]);
		register_rest_route($namespace, '/kixxl_wc_get_cart', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'kixxl_wc_get_cart_data'],
                'permission_callback' => '__return_true'
            ],
        ]);
		register_rest_route($namespace, '/kixxl_wc_update_cart_item', [
            [
                'methods' => 'POST',
                'callback' => [$this, 'kixxl_wc_update_cart_item'],
                'permission_callback' => '__return_true'
            ],
        ]);
        register_rest_route($namespace, '/get-encoded-resource-id', array(
            'methods' => 'GET',
            'callback' => [$this, 'get_encoded_resource_id'],
            'permission_callback' => '__return_true'
        ));
        register_rest_route($namespace, '/get-decoded-resource-id', array(
            'methods' => 'GET',
            'callback' => [$this, 'get_decoded_resource_id'],
            'permission_callback' => '__return_true'
        ));
		
		 register_rest_route('kixxl/v1', '/trigger-order-hook', [
        	'methods'  => 'GET',
            'callback' => [$this, 'kixxl_manual_order_hook_trigger'],
            'permission_callback' => '__return_true'
    	]);
    }
	
	
	
	function kixxl_manual_order_hook_trigger($request) 
	{
		$order_id = isset($_GET['order_id']) ? intval($_GET['order_id']) : 0;

		if (!$order_id) {
			return new WP_REST_Response(['error' => 'Missing or invalid order_id'], 400);
		}

		$order = wc_get_order($order_id);
		if (!$order) {
			return new WP_REST_Response(['error' => 'Order not found'], 404);
		}

		kixxl_wc_order_create($order_id, $order);

		return new WP_REST_Response(['message' => 'Order hook triggered successfully.'], 200);
	}
	
	public function kixxl_wc_get_cart_data()
    {
		$cart = WC()->cart->get_cart();
		$result = [];

		foreach ($cart as $item) {
			$result[] = [
				'edit_sheet' => isset($item['custom_data']['edit_sheet']) ? $item['custom_data']['edit_sheet'] : '',
				'key'        => $item['key'],
				'quantity'   => $item['quantity'],
			];
		}

		return new WP_REST_Response($result, 200);
    }
	
	public function kixxl_wc_update_cart_item(WP_REST_Request $request) 
	{
    	$params = $request->get_json_params();
		$sheetKey   = sanitize_text_field($params['sheetKey']);
		$quantity   = intval($params['quantity']);
		$properties = $params['properties'];

		if (!$sheetKey || !$quantity || !is_array($properties)) {
			return new WP_REST_Response(['error' => 'Missing or invalid parameters'], 400);
		}

		$cart = WC()->cart->get_cart();
		$found_key = null;

		foreach ($cart as $cart_item_key => $item) {
			if (!isset($item['custom_data']['edit_sheet'])) continue;

			$edit_sheet_url = $item['custom_data']['edit_sheet'];
			$url_parts = parse_url($edit_sheet_url);
			parse_str($url_parts['query'] ?? '', $query);

			if (isset($query['sheet_key']) && $query['sheet_key'] === $sheetKey) {
				$found_key = $cart_item_key;
				break;
			}
		}

		if (!$found_key) {
			return new WP_REST_Response(['error' => 'Cart item not found'], 404);
		}

		WC()->cart->set_quantity($found_key, $quantity, true);
		
		 // ✅ Update nested custom_data array
		if (!isset(WC()->cart->cart_contents[$found_key]['custom_data'])) {
			WC()->cart->cart_contents[$found_key]['custom_data'] = [];
		}

		foreach ($properties as $key => $value) {
			WC()->cart->cart_contents[$found_key]['custom_data'][$key] = sanitize_text_field($value);
		}

		WC()->session->set('cart', WC()->cart->get_cart_for_session());
    	WC()->cart->set_session(); // important
		WC()->cart->calculate_totals();

		return new WP_REST_Response(['success' => true, 'updated_key' => $found_key], 200);
	}

    /**
     * Store WooCommerce cart data to a temporary file.
     *
     * This function receives WooCommerce cart data, processes each item,
     * and saves the data to a temporary file for later retrieval.
     *
     * @return WP_REST_Response A response indicating success or failure, with the file path if successful.
     */
    public function kixxl_store_temp_cart_data()
    {
        $inputJSON = json_decode(file_get_contents('php://input'));
        $output = '';

        if (isset($inputJSON->items) && count($inputJSON->items) > 0) {
            foreach ($inputJSON->items as $index => $item) {
                {
                    $product_id = kixxl_wc_decode_resource_id($item->id);
                    $quantity = isset($item->quantity) ? (int)$item->quantity : 0;
                    if ($product_id > 0 && $quantity > 0) {
                        // Add product to the cart with custom data
                        $properties = $item->properties;
						
						$edit_sheet = isset($properties->edit_sheet) && $properties->edit_sheet !== ''
                        ? $properties->edit_sheet
                        : '';
						
						$edit_as_admin = isset($properties->_edit_as_admin) && $properties->_edit_as_admin !== ''
						? '|_edit_as_admin=' . $properties->_edit_as_admin
						: '';
						
						$sheet_size = isset($properties->sheet_size) && $properties->sheet_size !== ''
                    	? '|sheet_size=' . $properties->sheet_size
                    	: '';
						
						$has_overlapping_images = isset($properties->_has_overlapping_images) && $properties->_has_overlapping_images !== ''
                    	? '|_has_overlapping_images=' . $properties->_has_overlapping_images
                    	: '';
						
						$has_low_dpi_images = isset($properties->_has_low_dpi_images) && $properties->_has_low_dpi_images !== ''
                    	? '|_has_low_dpi_images=' . $properties->_has_low_dpi_images
                    	: '';
						
                        // Build base output
						 $output .= 'variantId=' . $product_id .
                        '|quantity=' . $item->quantity .
                        '|_app_name=' . $properties->_app_name .
                            '|_description=_' .
                            '|sheet_preview=' . $properties->sheet_preview .
                        '|price=' . $item->price .
                        '|_actual_gang_sheet=' . $properties->_actual_gang_sheet .
						$has_overlapping_images .
						$has_low_dpi_images .
						'|sheet_type=' . $properties->sheet_type .
						$sheet_size .
                            '|edit_sheet=' . $edit_sheet .
                            $edit_as_admin;

                        // Dynamically add keys with the prefix "extra_"
                        foreach ($properties as $key => $value) {
                            if (strpos($key, 'extra_') === 0 && $value !== '') {
                                $output .= '|' . $key . '=' . $value;
                            }
                        }

                        // Append comma at the end of this product
                        $output .= ',';
                    }
                }
            }
            $file_path = 'temp/' . $this->kixxl_wc_create_temp_cart_file($output);
            return new WP_REST_Response(array(
                'error' => false,
                'message' => 'File created successfully',
                'file' => $file_path
            ), 200);
        }

        return new WP_REST_Response(array(
            'error' => true,
            'message' => 'Error occurred',
        ), 500);
    }

    public function kixxl_wc_create_temp_cart_file($output)
    {
        if (!file_exists(plugin_dir_path(__FILE__) . '/temp')) {
            mkdir(plugin_dir_path(__FILE__) . '/temp', 0777, true);
        }
        $filename = time() . rand(10000, 99999999) . '.txt';
        $file = plugin_dir_path(__FILE__) . '/temp/' . $filename;
        $open = fopen($file, "a");
        $write = fputs($open, $output);
        fclose($open);
        return $filename;
    }

    public function kixxl_wc_logout_page()
    {
        ob_start();
        header('Content-Type: text/html');
        $html_file = plugin_dir_path(__FILE__) . 'templates/logout.html';
        echo file_get_contents($html_file);
    }

    public function kixxl_wc_login_page()
    {
        ob_start();
        header('Content-Type: text/html');
        $html_file = plugin_dir_path(__FILE__) . 'templates/login.html';
        echo file_get_contents($html_file);
    }

    public function kixxl_wc_logout_customer(WP_REST_Request $request)
    {
        // Log the user out
        // wp_logout();

        // Optionally clear all auth-related cookies (for extra safety)
        // wp_clear_auth_cookie();

        // Return response
        return rest_ensure_response([
            'success' => true,
            'message' => 'User logged out successfully.'
        ]);
    }

    public function kixxl_wc_login_customer(WP_REST_Request $request)
    {
        global $wpdb;

        $email = sanitize_email($request['email']);
        $password = sanitize_text_field($request['password']);
        $remember = $request['remember'];

        $cred['user_login'] = $email;
        $cred['user_password'] = $password;
        $cred['remember'] = $remember;

        $user = wp_signon($cred, false);

        if (is_wp_error($user)) {
            error_log('log in error......');
            error_log($user->get_error_message());
            return rest_ensure_response(array(
                'errors' => true,
                'message' => $user->get_error_message()
            ));
        } else {
            // Set auth cookie
            wp_set_auth_cookie($user->ID, $remember, false);

            // Retrieve the user data
            $userdata = get_userdata($user->ID);

            $username = $userdata->user_login;

            // Prepare response
            $response = array(
                'user' => $username,
                'customer_id' => kixxl_wc_generate_resource_id('customer', $userdata->ID),
                'errors' => false
            );

            return rest_ensure_response($response);
        }
    }

    public function kixxl_wc_register_page()
    {
        ob_start();
        header('Content-Type: text/html');
        $html_file = plugin_dir_path(__FILE__) . 'templates/register.html';
        echo file_get_contents($html_file);
    }


    /**
     * Searches for customer information based on the customer ID.
     *
     * This function retrieves customer information from WooCommerce based on
     * the provided customer ID. It returns the customer's email, first name,
     * last name, phone number, and default address.
     *
     * @param WP_REST_Request $request The request object containing the
     *     customer ID.
     *
     * @return WP_REST_Response|array The response object containing the
     *     customer information or an error message.
     */

    public function kixxl_wc_register_customer(WP_REST_Request $request)
    {
        $email = sanitize_email($request['email']);
        $password = sanitize_text_field($request['password']);
        $username = explode("@", $email)[0];

        $user = get_user_by('email', $email);
        $error = true;
        $message = '';
        if ($user) {
            $message = 'Account already exists. Please log in.';
        } else {
            $error = false;
            // Create new user
            $user_id = wp_create_user($username, $password, $email);
            if (!is_wp_error($user_id)) {
                $user = get_userdata($user_id);
                $user->set_role('customer');
            } else {
                $error = true;
                $message = 'Error during registration. Please try again';
                error_log($message);
                error_log($user_id->get_error_message());
            }
        }

        $sessionId = session_id();
        if (empty($sessionId)) {
            session_name('PHPSESSID');
            session_start();
        }

        // Log In
        if ($error) {
            return rest_ensure_response([
                "customer_id" => $user->ID,
                "errors" => true,
                "message" => $message,
            ]);
        } else {
            $cred['user_login'] = $email;
            $cred['user_password'] = $password;
            $cred['remember'] = true;
            wp_signon($cred, false);
        }

        return rest_ensure_response([
            "customer_id" => $user->ID,
            "errors" => false,
            "message" => $message,
        ]);
    }

    public function editor_load(WP_REST_Request $request)
    {
        global $config;
        header("Content-Type: text/html");

        $shop = kixxl_wc_get_shop($config);

        $variant = $_GET['variant'] ?? '';
        $product = $_GET['product'] ?? '';
        $quantity = $_GET['quantity'] ?? '';
        $customer = $_GET['customer'] ?? '';
        $woo_product_id = $_GET['wp'] ?? '';

        if (isset($variant) && $variant) {
            $variantId = kixxl_wc_generate_resource_id('variant', $variant);
        } else {
            $variantId = kixxl_wc_generate_resource_id('variant', $woo_product_id);
        }

        // Handle YayExtra data
        $extraRaw = $_GET['extra'] ?? '';
        $extraParam = '';

        if (!empty($extraRaw)) {
            // Decode URL-encoded input
            $decoded = rawurldecode($extraRaw);

            // Remove potential double escaping (e.g., {\"key\":\"value\"})
            $decoded = stripslashes($decoded);

            // Decode JSON into array
            $extraDecoded = json_decode($decoded, true);

            // Retry once more if the first decoding failed (handles nested JSON)
            if (json_last_error() !== JSON_ERROR_NONE) {
                $decodedRetry = stripslashes($decoded);
                $extraDecoded = json_decode($decodedRetry, true) ?? [];
            }

            if (!empty($extraDecoded)) {
                $extraParam = rawurlencode(json_encode($extraDecoded));
            }
        }

        // Base URL
        $baseUrl = get_site_url() . $config["PROXY_PREFIX"] . "/builder";

        // Query parameters
        $query = [
            'variant'   => $variantId,
            'product'   => $product,
            'quantity'  => $quantity,
            'customer'  => $customer,
            'local'     => 'true',
            'edit_link' => 'true',
        ];

        // Add YayExtra only if present
        if (!empty($extraParam)) {
            $query['extra'] = $extraParam;
        }

        // Build the final URL
        $url = $baseUrl . '?' . http_build_query($query, '', '&', PHP_QUERY_RFC3986);

        header('Location: ' . $url);
        exit();
    }
	
	public function get_shopify_customer($request)
	{
		$requestedCustomerId = $request->get_param('customerId');
		$customer_id = kixxl_wc_decode_resource_id($request->get_param('customerId'));
		
		if (empty($customer_id)) {
			error_log('Customer ID parameter is empty');
			return rest_ensure_response([
				"body" => [
					'customer' => [],
				],
				"errors" => true,
				"message" => "Customer ID parameter is empty"
			]);
		}

		$customer = null;
		$customerID = '';
		$customerEmail = '';

		global $wpdb;
		
		// if customer is guest customer so search in wc_lookup
		if(str_starts_with((string)$requestedCustomerId, '4418')) {
			$customer = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}wc_customer_lookup WHERE customer_id = %d", $customer_id));

			if ($customer) {
				$customerID = kixxl_wc_generate_resource_id('guest', $customer->customer_id);
				$customerEmail = $customer->email ?? '';
			}
		} else {
			$user = get_user_by('id', intval($customer_id));
			if ($user) {
				$customer = new WC_Customer($user->ID);
				$customerID = kixxl_wc_generate_resource_id('customer', $customer->get_id());
				$customerEmail = $customer->get_email();
			} else {
				error_log('Customer not found in user table');
				return rest_ensure_response([
					"body" => ["customer" => []],
					"errors" => true,
					"message" => 'Customer not found',
				]);
			}
		}
		
		return rest_ensure_response([
			"body" => [
					'customer' => [
						'id' => $customer_id,
						'legacyResourceId' => $customerID,
						'email' => $customerEmail,
					],
			],
			"errors" => false,
		]);
	}
	
	public function get_woo_customers($request)
	{
		$paged = $request['cursor'] ?? 1;
		$args = [
			'number'  => $request['limit'],
			'paged'   => $paged,
			'orderby' => 'ID',
			'order'   => 'DESC',
			'search'  => '*' . esc_attr($request['keyword']) . '*',
			'search_columns' => ['user_login', 'user_email', 'display_name'],
		];		

		$query = new WP_User_Query($args);	
		$users = $query->get_results();
		$total = $query->get_total();
		$total_pages = ceil($total / $request['limit']);

		$edges = array_map(function ($user) {
			$customer = new WC_Customer($user->ID);
			$address = [
				'address1'       => $customer->get_billing_address_1(),
				'city'           => $customer->get_billing_city(),
				'province'       => $customer->get_billing_state(),
				'zip'            => $customer->get_billing_postcode(),
				'country'        => $customer->get_billing_country(),
				'country_code'   => $customer->get_billing_country(),
				'province_code'  => $customer->get_billing_state(),
			];

			return [
				'cursor' => (string) $customer->get_id(),
				'node' => [
					'id'            => kixxl_wc_generate_resource_id('customer', $customer->get_id()),
					'email'         => $customer->get_email(),
					'firstName'     => $customer->get_first_name(),
					'lastName'      => $customer->get_last_name(),
					'phone'      	=> $customer->get_billing_phone(),
					'defaultAddress'=> $address,
				],
			];
		}, $users);
		
			return rest_ensure_response([
			  	'body' => [
					'data' => [
						'customers' => [
							'edges' => $edges,
							'pageInfo' => [
								'hasNextPage' => $paged < $total_pages,
								'endCursor'   => $paged < $total_pages ? $paged + 1 : null,
							],
						],
					],
				],
				"errors" => false,
		]);
	}
	
	
	public function search_customer($request)
	{
		$email = $request->get_param('email') ?? '';
		$requestedCustomerId = $request->get_param('customerId');
		$customer_id = kixxl_wc_decode_resource_id($request->get_param('customerId'));
		
		if (empty($email) && empty($customer_id)) {
			error_log('Email or Customer ID parameter is empty');
			return rest_ensure_response([
				"body" => [
					'customer' => [],
				],
				"errors" => true,
				"message" => "Email or Customer ID parameter is empty"
			]);
		}

		$customer = null;
		$customerID = '';
		$customerEmail = '';
		$customerFirstName = '';
		$customerLastName = '';
		$customerPhone = '';

		global $wpdb;
		
		// if customer is guest customer so search in wc_lookup
		if(str_starts_with((string)$requestedCustomerId, '4418')) {
			$customer = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$wpdb->prefix}wc_customer_lookup WHERE customer_id = %d", $customer_id));

			if ($customer) {
				$customerID = kixxl_wc_generate_resource_id('guest', $customer->customer_id);
				$customerEmail = $customer->email ?? '';
				$customerFirstName = $customer->first_name ?? '';
				$customerLastName = $customer->last_name ?? '';
				$customerPhone = $customer->phone ?? '';
			}
		}
		// Registered customer 
		elseif(!empty($customer_id)) {
			$user = get_user_by('id', intval($customer_id));
			if ($user) {
				$customer = new WC_Customer($user->ID);
				$customerID = kixxl_wc_generate_resource_id('customer', $customer->get_id());
				$customerEmail = $customer->get_email();
				$customerFirstName = $customer->get_billing_first_name();
				$customerLastName = $customer->get_billing_last_name();
				$customerPhone = $customer->get_billing_phone();
			} else {
				error_log('Customer not found in user table');
				return rest_ensure_response([
					"body" => ["customer" => []],
					"errors" => true,
					"message" => 'Customer not found',
				]);
			}
		}
		// If email is provided and customer_id is not, search for the customer by email
		elseif (!empty($email)) {
			$max_attempts = 50;
			$attempts = 0;

			while ($attempts < $max_attempts) {
				$customer = $wpdb->get_row($wpdb->prepare("
					SELECT * FROM {$wpdb->prefix}wc_customer_lookup 
					WHERE email = %s
				", sanitize_email($email)));
				if (!empty($customer)) {
					$customerID = kixxl_wc_generate_resource_id('guest', $customer->customer_id);
					$customerEmail = $customer->email ?? '';
					$customerFirstName = $customer->first_name ?? '';
					$customerLastName = $customer->last_name ?? '';
					$customerPhone = $customer->phone ?? '';

					// Get last order of customer
					if (!empty($customerEmail)) {
						$query = new WC_Order_Query([
							'billing_email' => $email ?? $customerEmail,
							'orderby'       => 'date',
							'order'         => 'DESC',
							'limit'         => 1,
						]);
						$orders = $query->get_orders();

						if (!empty($orders)) {
							$order = $orders[0];
							$phoneFromOrder = $order->get_billing_phone();
							if (!empty($phoneFromOrder)) {
								$customerPhone = $phoneFromOrder;
							}
						}
					}

					break; // Exit the loop if a customer is found
				}
				$attempts++;
				usleep(3000000); // Sleep for 0.5 seconds before retrying
			}
		}

		if (empty($customer)) {
			return rest_ensure_response([
				"body" => [
					'customer' => [],
				],
				"errors" => true,
			]);
		}
		
		$last_order = null;
		if ($customerID && !empty($customerEmail)) {
			$query = new WC_Order_Query([
				'billing_email' => $email ?? $customerEmail,
				'orderby'       => 'date',
				'order'         => 'DESC',
				'limit'         => 1,
			]);

			$orders = $query->get_orders();

			if (!empty($orders)) {
				$order = $orders[0];

				// Line items
				$line_items = [];
				foreach ($order->get_items() as $item) {
					$product = $item->get_product();
					$image_url = $product ? wp_get_attachment_url($product->get_image_id()) : null;
					$variant_title = $product ? $product->get_attribute('pa_size') : null; // Adjust based on attribute name

					$line_items[] = [
						'id' => 'gid://woocommerce/LineItem/' . $item->get_id(),
						'name' => $item->get_name(),
						'image' => [
							'url' => $image_url,
						],
						'title' => $product ? $product->get_name() : $item->get_name(),
						'variant' => [
							'title' => $variant_title ?? '',
						],
						'quantity' => $item->get_quantity(),
						'originalTotalSet' => [
							'presentmentMoney' => [
								'amount' => wc_format_decimal($item->get_total(), 2),
								'currencyCode' => $order->get_currency(),
							],
						],
						'originalUnitPriceSet' => [
							'presentmentMoney' => [
								'amount' => wc_format_decimal($item->get_total() / max(1, $item->get_quantity()), 2),
								'currencyCode' => $order->get_currency(),
							],
						],
					];
				}

				// Final last order data
                $last_order = [
                    'nodes' => [
                        [
                            'canMarkAsPaid' => ($order instanceof WC_Order) ? !$order->is_paid() : false,
                            'createdAt'     => ($order instanceof WC_Order && $order->get_date_created())
                                ? $order->get_date_created()->date('c')
                                : null,
                            'currencyCode'  => ($order instanceof WC_Order && $order->get_currency())
                                ? $order->get_currency()
                                : 'USD',
                            'fullyPaid'     => ($order instanceof WC_Order) ? $order->is_paid() : true,
                            'lineItems'     => [
                                'nodes' => $line_items ?? [],
                            ],
                        ],
                    ],
                ];
			}
		}

			return rest_ensure_response([
			"body" => [
				"data" => [
					'customer' => [
						'id' => $customerID,
						'legacyResourceId' => $customerID,
						'email' => $customerEmail,
						'firstName' => $customerFirstName,
						'lastName' => $customerLastName,
						'phone' => $customerPhone,
						'orders' => $last_order,
						'defaultAddress' => [
							'city' => isset($customer->billing->city) && $customer->billing->city !== '' ? $customer->billing->city : ($customer->city ?? ''),
							'country' => isset($customer->billing->country) && $customer->billing->country !== '' ? $customer->billing->country : ($customer->country ?? ''),
						],
					],
				],
			],
			"errors" => false,
		]);
	}
	
	
    function kixxl_authenticate(WP_REST_Request $request)
    {
        $tokenId = $request['tokenId'];
        $token = $request['token'];
        if (!isset($tokenId) || !isset($token)) {
            return new WP_Error('missing_params', 'TokenId and token are required', array('status' => 400));
        }

        $filename = $tokenId;
        $content = $token;

        $upload_dir = wp_upload_dir();
        $directory = $upload_dir['basedir'] . '/tmp/cache/kixxl';

        if (!file_exists($directory)) {
            wp_mkdir_p($directory);
        }

        $file_path = $directory . '/' . $filename;

        if (file_put_contents($file_path, $content) === false) {
            return new WP_REST_Response(array('error' => true, 'message' => 'Failed to write to file'), 500);
        }


        return new WP_REST_Response(array(
            'error' => false,
            'message' => 'File created successfully',
            'file' => $file_path
        ), 200);
    }

    /**
     * Handles the Bearer token from the request, decodes it, and returns a new token.
     *
     * @param WP_REST_Request $request
     * @return WP_REST_Response
     */
    function kixxl_wc_get_token(WP_REST_Request $request)
    {
        $shop = $request->get_param('store');  // Get store from query parameter
        $token = $request->get_param('token');  // Get token from query parameter

        // Check if the store is provided
        if (empty($shop)) {
            return new WP_REST_Response(array('error' => 'Store not provided'), 400);
        }

        // Check if the token is provided
        if (empty($token)) {
            return new WP_REST_Response(array('error' => 'Token not provided'), 400);
        }

        try {
            // Check if shop is valid
            $shopId = get_option('_kixxl_' . $shop . '_shop_id');
            $shopPassw = get_option('_kixxl_' . $shop . '_shop_password');
            if (empty($shopId) || empty($shopPassw)) {
                return new WP_REST_Response(array('error' => 'Invalid Store provided'), 400);
            }

            // Check if token is valid
            if (!kixxl_wc_validate_token($token, $shopPassw)) {
                return new WP_REST_Response(array('error' => 'Invalid token provided'), 400);
            }

            // Generate new JWT token
            $payload = [
                "user_id" => $shopId,
                "username" => $shop,
                "exp" => time() + 150, // Token expiration time (2.5 min)
            ];
            $tokenNew = kixxl_wc_create_jwt_token($payload, $shopPassw);

            // Return the new JWT token
            return new WP_REST_Response(array('token' => $tokenNew), 200);

        } catch (Exception $e) {
            return new WP_REST_Response(array('error' => 'Invalid token: ' . $e->getMessage()), 400);
        }
    }

    function fetch_shop_detail_from_api(WP_REST_Request $request)
    {
        try {
            $user = get_users(array(
                'role' => 'administrator',
                'number' => 1 // Get only one admin user
            ))[0];

            $shopDetail = [];

            // Format data
            $shopDetail['name'] = get_bloginfo('name');
            $shopDetail['email'] = $user->data->user_email;
            $shopDetail['currencyCode'] = get_woocommerce_currency();

            // Get timezone
            $shopDetail['ianaTimezone'] = get_option('timezone_string') ?: 'UTC';

            $response = [
                'errors' => false,
                'status' => 200,
                'body' => [
                    'data' => [
                        'shop' => $shopDetail,
                        'shopLocales' => [['locale' => substr(get_locale(), 0, 2)]]
                    ],
                ],
            ];

            return rest_ensure_response($response);
        } catch (Throwable $exception) {
            return rest_ensure_response([
                'errors' => true,
                'status' => 500,
                'exception' => [
                    'message' => $exception->getMessage(),
                    'code' => $exception->getCode(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString()
                ]
            ]);
        }
    }

    /**
     * This function is used to generate resource id
     *
     *
     * @param WP_REST_Request $request
     *
     * @return mixed
     */
    public function get_encoded_resource_id(WP_REST_Request $request)
    {
        try {
            return rest_ensure_response([
                "errors" => false,
                "response" => [],
                "status" => 200,
                "body" => [
                    'resourceId' => kixxl_wc_generate_resource_id($request['resource_type'],
                        $request['resource_id'])
                ]
            ]);
        } catch (Throwable $exception) {
            return rest_ensure_response([
                'errors' => true,
                'status' => 500,
                'exception' => [
                    'message' => $exception->getMessage(),
                    'code' => $exception->getCode(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString(),
                ]
            ]);
        }
    }

    /**
     * This function is used to generate decoded resource id
     *
     *
     * @param WP_REST_Request $request
     *
     * @return mixed
     */
    public function get_decoded_resource_id(WP_REST_Request $request)
    {
        try {
            return rest_ensure_response([
                "errors" => false,
                "response" => [],
                "status" => 200,
                "body" => [
                    'resourceId' => kixxl_wc_decode_resource_id($request['resource_id'])
                ]
            ]);
        } catch (Throwable $exception) {
            return rest_ensure_response([
                'errors' => true,
                'status' => 500,
                'exception' => [
                    'message' => $exception->getMessage(),
                    'code' => $exception->getCode(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString(),
                ]
            ]);
        }
    }
}
