<?php

/**
 * Order_Details_Controller class.
 *
 * This class handles the registration of custom REST API routes for retrieving
 * order details in a WooCommerce store. The routes are registered under the
 * namespace 'kwc/v1'.
 */
class Order_Details_Controller extends WP_REST_Controller
{
	public function __construct() {
		$this->init_hooks();
	}

    /**
     * Registers the routes for the custom REST API endpoints.
     *
     * This function registers a route for retrieving order details based on
     * the order ID.
     */
    public function register_routes()
    {
        $version = '1';
        $namespace = 'kwc/v' . $version;
        $base = 'get-order';

        register_rest_route($namespace, '/' . $base, [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_order'],
                'permission_callback' => '__return_true',
                // Adjust as needed for security

            ],
        ]);

        register_rest_route($namespace, '/search-order', [
            'methods' => 'GET',
            'callback' => [$this, 'search_order'],
            'permission_callback' => '__return_true', // Change this to check permissions if needed
        ]);

        register_rest_route($namespace, '/create-draft-order', [
            'methods' => 'POST',
            'callback' => [$this, 'create_draft_order'],
            'permission_callback' => '__return_true', // Change this to check permissions if needed
        ]);
		
		register_rest_route($namespace, '/send-invoice', [
            'methods' => 'POST',
            'callback' => [$this, 'send_invoice'],
            'permission_callback' => '__return_true', // Change this to check permissions if needed
        ]);
    }

    /**
     * Retrieves order details based on the provided order ID.
     *
     * This function fetches order details from WooCommerce based on the
     * provided order ID. It returns information such as order number,
     * financial status, total price, customer details, shipping address,
     * shipping lines, and line items.
     *
     * @param WP_REST_Request $request The request object containing the order
     *     ID.
     *
     * @return WP_REST_Response|array The response object containing the order
     *     details or an error message.
     */
    public function get_order($request)
    {
        $order_id = kixxl_wc_decode_resource_id($request['orderId']);
        // Fetch order details
        $order = wc_get_order($order_id);
        if (!$order) {
            return new WP_Error('order_not_found', __('Order not found.'), ['status' => 404]);
        }
        $shipping = $order->get_address('shipping');
        $shipping_lines = $order->get_shipping_methods();
        $shipping_lines_data = [];
        foreach ($shipping_lines as $shipping_line) {
			$method_id = $shipping_line->get_method_id();
			if($method_id == 'pickup_location') {
				$shipping = null;
			}
			
			 // Get base data
			$data = $shipping_line->get_data();
			$data['title'] = $shipping_line->get_name();
			$shipping_lines_data[] = $data;
        }
        // Get order items
        $order_items = $order->get_items();

        $items_data = [];
        $custom_tags='';

        foreach ($order_items as $item_id => $item) {
            if (!empty($item->get_meta('tags'))) {
                $custom_tags = $item->get_meta('tags');
            }
            $quantity = $item->get_quantity();
            $subtotal = $order->get_line_subtotal($item);
            $price_per_unit = $quantity > 0 ? $subtotal / $quantity : 0;
			
			
			$product = $item->get_product();
			if ($product && $product->is_type('variation')) {
				$parent_id = $product->get_parent_id();
				$name = get_the_title($parent_id);
			} else {
				$name = $item->get_name();
			}
			
			$attributes = [];
			if ($product && $product->is_type('variation')) {
				$variation_data = $product->get_attributes();

				foreach ($variation_data as $key => $value) {
					// Clean taxonomy attribute keys (e.g., "pa_color" => "Color")
					$taxonomy = str_replace('attribute_', '', $key);
					if (taxonomy_exists($taxonomy)) {
						$term = get_term_by('slug', $value, $taxonomy);
						$attributes[] = $term ? $term->name : $value;
					} else {
						$attributes[] = $value;
					}
				}
			}
			$full_name = !empty($attributes) ? $name . ' - ' . implode(', ', $attributes) : $name;
			
            $items_data[] = [
                'id' => $item->get_id(),
                'product_id' => kixxl_wc_generate_resource_id('product', $item->get_product_id()),
                'name' => $full_name,
                'quantity' => $quantity,
                'variant_id' => kixxl_wc_generate_resource_id('variant', $item->get_variation_id()),
                'price' => $price_per_unit,
                'price_set' => [
                    'shop_money' => [
                        'amount' => $price_per_unit,
                    ]
                ],
                'sku' => $item->get_product() ? $item->get_product()->get_sku() : '',
                'properties' => [
                    [
                        "name" => "_app_name",
                        "value" => $item->get_meta('_app_name'),
                    ],
                    [
                        "name" => "sheet_preview",
                        "value" => $item->get_meta('sheet_preview'),
                    ],
                    [
                        "name" => "edit_sheet",
                        "value" => $item->get_meta('edit_sheet'),
                    ],
                    [
                        "name" => "sheet_type",
                        "value" => $item->get_meta('sheet_type'),
                    ],
                    [
                        "name" => "_actual_gang_sheet",
                        "value" => $item->get_meta('_actual_gang_sheet'),
                    ],
                ]
            ];
        }
        $orderData = $order->get_data();

        // Get order data
        $order_data = [
            'id' => kixxl_wc_generate_resource_id('order', $orderData['id']),
            'order_number' => $order_id,
            'name' => $orderData['number'],
            'financial_status' => $orderData['status'],
            'currency' => $orderData['currency'],
            'current_total_price' => $orderData['total'],
            'current_total_price_set' => [
                'shop_money' => [
                    'amount' => $orderData['total'],
                    'currency_code' => $orderData['currency']
                ]
            ],
            'price_set' => [
                'shop_money' => [
                    'amount' => $orderData['total'],
                    'currency_code' => $orderData['currency']
                ]
            ],
            'price_without_tax' => $orderData['total'] - $orderData['total_tax'],
            'note' => $orderData['customer_note'],
            'created_at' => $orderData['date_created']->date('Y-m-d H:i:s'),
            'tags' => $custom_tags,
            "properties" => [],
            "customer" => [
                "id" => kixxl_wc_generate_resource_id('customer', $orderData['customer_id']),
                "email" => $orderData['billing']['email'],
                "first_name" => $orderData['billing']['first_name'],
                "last_name" => $orderData['billing']['last_name'],
                "state" => "enabled",
                "note" => NULL,
                "verified_email" => TRUE,
                "multipass_identifier" => NULL,
                "tax_exempt" => FALSE,
                "phone" => $orderData['billing']['phone'],
                "currency" => $orderData['currency'],
                "default_address" => [
                    "id" => uniqid(),
                    "customer_id" => kixxl_wc_generate_resource_id('customer', $orderData['customer_id']),
                    "first_name" => $orderData['billing']['first_name'],
                    "last_name" => $orderData['billing']['last_name'],
                    "address1" => $orderData['billing']['address_1'],
                    "address2" => $orderData['billing']['address_2'],
                    "city" => $orderData['billing']['city'],
                    "province" => $orderData['billing']['state'],
                    "country" => $orderData['billing']['country'],
                    "zip" => $orderData['billing']['postcode'],
                    "phone" => $orderData['billing']['phone'],
                    "country_code" => $orderData['billing']['country'],
                    "default" => TRUE,
                ],
            ],
            "shipping_address" => $shipping,
            "shipping_lines" => $shipping_lines_data,
            "line_items" => $items_data,

        ];

        $response = [
            "body" => [
                'order' => $order_data,
            ],
            "errors" => false,
        ];

        return rest_ensure_response($response);
    }

    public function search_order($request)
    {
        // Fetch order details
        $order = wc_get_order($request['orderNumber']);
        if (!$order) {
            return new WP_Error('order_not_found', __('Order not found.'), ['status' => 404]);
        }
        // Get order items
        $order_items = $order->get_items();

        $items_data = [];

        foreach ($order_items as $item) {
            $items_data['nodes'][] = [
                'id' => $item->get_id(),
                'product_id' => kixxl_wc_generate_resource_id('product', $item->get_product_id()),
                'name' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'variant_id' => kixxl_wc_generate_resource_id('variant', $item->get_variation_id()),
                'originalUnitPriceSet' => [
                    'shopMoney' => [
                        'amount' => $order->get_line_subtotal($item, FALSE, FALSE),
                    ]
                ],
                'sku' => $item->get_product() ? $item->get_product()->get_sku() : '',
                'customAttributes' => [
                    [
                        "key" => "_app_name",
                        "value" => $item->get_meta('_app_name'),
                    ],
                    [
                        "key" => "sheet_preview",
                        "value" => $item->get_meta('sheet_preview'),
                    ],
                    [
                        "key" => "sheet_type",
                        "value" => $item->get_meta('sheet_type'),
                    ],
                    [
                        "key" => "_description",
                        "value" => $item->get_meta('_description')
                    ],
                    [
                        "key" => "_actual_gang_sheet",
                        "value" => $item->get_meta('_actual_gang_sheet'),
                    ],
                    [
                        "key" => "sheet_size",
                        "value" => $item->get_meta('sheet_size')
                    ],
                    [
                        "key" => "edit_sheet",
                        "value" => $item->get_meta('edit_sheet')
                    ],
                ],
                'product' => [
                    'legacyResourceId' => kixxl_wc_generate_resource_id('product', $item->get_product_id()),
                    'title' => $item->get_name(),
                ],
                'variant' => [
                    'legacyResourceId' => kixxl_wc_generate_resource_id('variant', $item->get_variation_id()),
                ]
            ];
        }
        $orderData = $order->get_data();
		
		if(empty($orderData['customer_id'])) {
			$email =  $orderData['billing']['email'] ?? '';
			$customerData = find_customer_by_email($email);
			$customerId = kixxl_wc_generate_resource_id('guest', $customerData->data->customer_id ?? 0);
		} else {
			$customerId = kixxl_wc_generate_resource_id('customer',$orderData['customer_id']);
		}
		
        // Get order data
        $order_data['nodes'] = [[
            'id' => kixxl_wc_generate_resource_id('order', $orderData['id']),
            'name' => $orderData['number'],
            "customer" => [
                "id" => $customerId,
                "legacyResourceId" => $customerId,
            ],
            'subtotalPriceSet' => [
                'shopMoney' => [
                    'amount' => $orderData['total'],
                    'currency_code' => $orderData['currency']
                ]
            ],
            "lineItems" => $items_data,

        ]];

        $response = [
            "body" => [
                "data" => [
                    "orders" => $order_data
                ]
            ],
            "errors" => false,
        ];

        return rest_ensure_response($response);
    }

    public function create_draft_order(WP_REST_Request $request)
    {
		$params = $request->get_json_params();
		$tags = isset($params['data']['draft_order']['tags']) ? sanitize_text_field($params['data']['draft_order']['tags']) : '';
		
		if (!isset($params['data']['draft_order']['line_items'])) {
        	return new WP_REST_Response(['errors' => true, 'message' => 'Missing line_items.'], 400);
    	}
		
		$lineItems = $params['data']['draft_order']['line_items'];
		
		// Handle customer if passed
		$customerId = $params['data']['draft_order']['customer']['id'] ?? null;
		$orderArgs = ['status' => 'pending'];
		if ($customerId && get_user_by('id', $customerId)) {
			$orderArgs['customer_id'] = $customerId;
		}

		$order = wc_create_order($orderArgs);
		if (!empty($tags)) {
    		$order->add_order_note($tags, true);
		}
    	$mappedItems = [];
    	$total = 0;
		
		foreach ($lineItems as $item) 
		{
			$quantity = isset($item['quantity']) ? intval($item['quantity']) : 1;
			$price    = isset($item['price']) ? floatval($item['price']) : 0.0;
			$name     = isset($item['title']) ? sanitize_text_field($item['title']) : 'Custom Item';

			// Use the correct class to allow quantity/price setting
			$orderItem = new WC_Order_Item_Product();
			$orderItem->set_name($name);
			$orderItem->set_quantity($quantity);
			$orderItem->set_total($price * $quantity);
			$orderItem->set_subtotal($price * $quantity);
			$orderItem->set_tax_class('');
			
			// Add custom properties as meta data
			if (!empty($item['properties']) && is_array($item['properties'])) {
				foreach ($item['properties'] as $property) {
					$propName = sanitize_text_field($property['name'] ?? '');
					$propValue = sanitize_text_field($property['value'] ?? '');
					if (!empty($propName)) {
						$orderItem->add_meta_data($propName, $propValue, true);
					}
				}
			}
			$order->add_item($orderItem);

			$mappedItems[] = [
				'id'                => rand(10000000000000, 99999999999999),
				'variant_id'        => null,
				'product_id'        => null,
				'title'             => $name,
				'variant_title'     => null,
				'sku'               => null,
				'vendor'            => null,
				'quantity'          => $quantity,
				'requires_shipping' => false,
				'taxable'           => true,
				'gift_card'         => false,
				'fulfillment_service' => 'manual',
				'grams'             => 0,
				'tax_lines'         => [],
				'applied_discount'  => null,
				'name'              => $name,
				'properties'        => [],
				'custom'            => true,
				'price'             => number_format($price, 2, '.', ''),
			];

			$total += $price * $quantity;
		}

		$order->calculate_totals();
		$tax_lines = [];
		foreach ($order->get_items('tax') as $taxItem) {
			$tax_lines[] = [
				'rate'  => floatval($taxItem->get_rate_percent()) / 100,
				'title' => $taxItem->get_label(),
				'price' => wc_format_decimal($taxItem->get_tax_total(), 2),
			];
		}

		$orderId    = $order->get_id();
		$paymentUrl = $order->get_checkout_payment_url();
		
		 $response = [
			'errors' => false,
			'status' => 201,
			'body' => [
				'draft_order' => [
					'id' => $orderId,
					'currency' => get_woocommerce_currency(),
					'status' => 'open',
					'line_items' => $mappedItems,
					'invoice_url' => $paymentUrl,
					'total_price' => wc_format_decimal($order->get_total(), 2),
					'subtotal_price' => wc_format_decimal($order->get_subtotal(), 2),
					'total_tax' => wc_format_decimal($order->get_total_tax(), 2),
					'tax_lines' => $tax_lines,
					'tags' => $tags,
				]
			]
		];

        return rest_ensure_response($response);
    }
	
	public function send_invoice($request)
	{
		$params    = $request->get_json_params();
		$order_id  = isset($params['orderId']) ? intval($params['orderId']) : 0;
		$emailData = $params['data']['draft_order_invoice'] ?? [];

		if (!$order_id) {
			return new WP_REST_Response([
				'errors' => true,
				'message' => 'Missing order ID',
			], 400);
		}

		$order = wc_get_order($order_id);
		if (!$order) {
			return new WP_REST_Response([
				'errors' => true,
				'message' => 'Order not found',
			], 404);
		}
		
		 // Set custom subject and recipient
		if (!empty($emailData['subject'])) {
			$order->update_meta_data('_custom_invoice_subject', sanitize_text_field($emailData['subject']));
		}

		if (!empty($emailData['to'])) {
			$order->update_meta_data('_custom_invoice_to', sanitize_email($emailData['to']));
		}
		
		$order->save();
		
		// Trigger the invoice email here
		WC()->mailer()->emails['WC_Email_Customer_Invoice']->trigger($order_id);
		return new WP_REST_Response([
			'errors' => false,
			'status' => 201,
			'message' => 'Invoice sent successfully',
		], 200);
	}
	
	
	public function init_hooks() 
	{
		add_filter('woocommerce_email_subject_customer_invoice', [$this, 'custom_invoice_subject'], 10, 2);
		add_filter('woocommerce_email_recipient_customer_invoice', [$this, 'custom_invoice_recipient'], 10, 2);
	}
	
	// Modify subject
	public function custom_invoice_subject($subject, $order) 
	{
		$custom = $order->get_meta('_custom_invoice_subject');
		return $custom ?: $subject;
	}

	// Modify recipient
	public function custom_invoice_recipient($recipient, $order) 
	{
		$custom = $order->get_meta('_custom_invoice_to');
		return $custom ?: $recipient;
	}

}
