<?php

/**
 * Product_Details_Controller class.
 *
 * This class handles the registration of custom REST API routes for retrieving
 * product details and searching products in a WooCommerce store. The routes
 * are registered under the namespace 'kwc/v1'.
 */
class Product_Details_Controller extends WP_REST_Controller
{
    protected $statuses = [
        "publish" => "ACTIVE"
    ];

    /**
     * Registers the routes for the custom REST API endpoints.
     *
     * This function registers two routes:
     * 1. A route for retrieving product details based on an encrypted product
     * ID and user ID.
     * 2. A route for searching products based on a search query, excluding
     * specific product IDs, and a user ID.
     */
    public function register_routes()
    {
        $version = '1';
        $namespace = 'kwc/v' . $version;

        register_rest_route($namespace, '/' . 'get-product-using-graph-ql', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_product_using_graphql'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'get-collection-by-name', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_collection_by_name'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'delete-product', [
            [
                'methods' => 'DELETE',
                'callback' => [$this, 'delete_product'],
                'permission_callback' => '__return_true',
            ],
        ]);

        // search product route
        register_rest_route($namespace, '/' . 'search-product-using-graph-ql', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'search_product_using_graphql'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'search-product-by-id-using-graph-ql', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'search_product_by_id_using_graphql'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'get-product-meta-fields', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_product_meta_fields'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'get-locations', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_locations'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'get-publications', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'get_publications'],
                'permission_callback' => '__return_true',
            ],
        ]);

        register_rest_route($namespace, '/' . 'send-invoice', [
            [
                'methods' => 'GET',
                'callback' => [$this, 'send_invoice'],
                'permission_callback' => '__return_true',
            ],
        ]);
    }

    public function get_collection_by_name()
    {
        $response = [
            'body' => [],
            'errors' => false,
        ];

        return rest_ensure_response($response);
    }

    public function delete_product($request)
    {
        try {
            $product_id = kixxl_wc_decode_resource_id($request['productId']);

            $product = wc_get_product($product_id);

            if ($product) {
                if ($product->is_type('variable')) {
                    // Delete all children
                    foreach ($product->get_children() as $variant_id) {
                        $variant = wc_get_product($variant_id);
                        $variant->delete(true);
                    }
                }

                // true is for permanent delete
                $product->delete(true);
            }

            $deletedProductId = $product ? $request['productId'] : null;

            return rest_ensure_response([
                'errors' => false,
                'response' => [],
                'status' => 200,
                'body' => [
                    'data' => [
                        'productDelete' => [
                            'deletedProductId' => $deletedProductId,
                            'userErrors' => []
                        ]
                    ]
                ]
            ]);
        } catch (Throwable $exception) {
            return $this->error_response($exception);
        }
    }

    /**
     * Retrieves product details based on the encrypted product ID and user ID.
     *
     * This function decrypts the product ID using the user ID and retrieves
     * the product details from WooCommerce. It returns the product
     * information, including title, status, image, and variants.
     *
     * @param WP_REST_Request $request The request object containing the
     *     encrypted product ID and user ID.
     *
     * @return WP_REST_Response|WP_Error The response object containing the
     *     product details or an error message.
     */
    public function get_product_using_graphql($request)
    {
        $encryptedProductId = $request['productId'];
        $userId = $request['userId'];

        $productId = kixxl_wc_decode_resource_id($encryptedProductId);

        $product = wc_get_product($productId);
        if (!$product) {
            return new WP_Error('product_not_found', __('Product not found.'), ['status' => 404]);
        }

        $product_data = new stdClass();
        $product_data->id = $encryptedProductId;
        $product_data->title = html_entity_decode($product->get_name(), ENT_QUOTES, 'UTF-8');
        $product_data->status = $this->statuses[$product->get_status()] ?? '';

        $image_id = $product->get_image_id();
        $image_url = wp_get_attachment_image_url($image_id, 'full');
        $product_data->featuredImage = (object)['url' => $image_url];

        $product_data->productType = "";

        $variants_data = [];
        if ($product->is_type('variable')) {
            $variations = $product->get_available_variations();
            foreach ($variations as $key => $variation) {
                $variationTitleConcat = "";
                $selectedOptions = [];
                foreach ($variation['attributes'] as $attr_key => $value) {
                    $attribute = substr($attr_key, strlen("attribute_"));
                    // Get the actual label WooCommerce uses for display
					$attribute_label = wc_attribute_label($attribute);
					$selectedOptions[] = [
						"name" => $attribute_label,
						"value" => $value,
					];
                    if (!empty($variationTitleConcat)) {
                        $variationTitleConcat .= ' / ';
                    }
                    $variationTitleConcat .= $value;
                }

                $variantId = kixxl_wc_generate_resource_id('variant', $variation['variation_id']);
				$is_taxable = get_post_meta($variation['variation_id'], '_tax_status', true) === 'taxable';
				
                $variants_data[] = (object)[
                    'legacyResourceId' => $variantId,
                    'title' => $variationTitleConcat,
                    'price' => empty($variation['display_price']) ? 0.00 : (float)$variation['display_price'],
                    'sku' => $variation['sku'],
                    'displayName' => $product_data->title . ' - ' . $variation['display_price'] . '/' . $variationTitleConcat,
                    'position' => $key + 1,
                    'selectedOptions' => $selectedOptions,
					'taxable' => $is_taxable
                ];
            }
        } else {
            $selectedOptions = [["name" => 'Title', "value" => 'small']];
            $variants_data[] = (object)[
                'legacyResourceId' => kixxl_wc_generate_resource_id('variant', $productId),
                'title' => 'Default Title',
                'price' => empty($product->get_price()) ? 0.00 : (float)$product->get_price(),
                'sku' => $product->get_sku(),
                'displayName' => $product_data->title . ' - ' . $product->get_price(),
                'position' => 1,
                'selectedOptions' => $selectedOptions,
            ];
        }
		
		// Extract unique attribute names only
		$options = [];
		$attributes = $product->get_attributes();

		foreach ($attributes as $attribute_name => $attribute) {
			$label = wc_attribute_label($attribute_name);
			$options[] = [ 'name' => $label ];
		}

		$product_data->options = $options;
        $product_data->variants = (object)['nodes' => $variants_data];

        $response = (object)[
            'body' => (object)[
                'data' => (object)['product' => $product_data],
            ],
            'errors' => false,
        ];

        return (object)rest_ensure_response($response);
    }

    /**
     * Searches for products based on the search query and excluding specific
     * product IDs.
     *
     * This function performs a search query for products in WooCommerce and
     * excludes products based on the provided product IDs. It returns the
     * search results, including product details, images, and variants.
     *
     * @param WP_REST_Request $request The request object containing the search
     *     query, product IDs to exclude, and user ID.
     *
     * @return WP_REST_Response|WP_Error The response object containing the
     *     search results or an error message.
     */
    public function search_product_using_graphql($request)
    {
        try {
            $search_query = $request['title'];
            $productIds_to_exclude = $request['productIds'];
            $userId = $request['userId'];
            if (empty($search_query)) {
                return rest_ensure_response((object)[
                    'errors' => (object)['message' => __('Search query is required.')],
                    'status' => 400
                ]);
            }

            $product_query_args = [
                'status' => 'publish',
                'limit' => -1,
            ];

            if (!empty($search_query)) {
                $product_query_args['s'] = $search_query;
            }
            $pattern = '/^[a-zA-Z0-9_-]{1,43}$/'; // Allows 1 to 43 characters
            if (!empty($productIds_to_exclude)) {
                //$productIds_to_exclude = json_decode($productIds_to_exclude);
                $decryptedProductIds = array_map(function ($value) use ($pattern, $userId) {
                    return preg_match($pattern, $value) ? kixxl_wc_decode_resource_id($value) : $value;
                }, $productIds_to_exclude);

                $product_query_args['exclude'] = $decryptedProductIds;
            }

            $products_query = new WC_Product_Query($product_query_args);
            $products = $products_query->get_products();

            if (empty($products)) {
                return rest_ensure_response((object)[
                    "body" => (object)[
                        "data" => (object)[
                            "products" => (object)[
                                "nodes" => (object)[],
                            ],
                        ],
                    ],
                    "errors" => false,
                ]);
            }

            $product_data = [];
            foreach ($products as $product) {
                $image_id = $product->get_image_id();
                $image_url = wp_get_attachment_image_url($image_id, 'full');
                $selectedOptions = [];
                $variants_data = [];
                $hasOnlyDefaultVariant = false;
                if ($product->is_type('variable')) {
                    $variations = $product->get_available_variations();
                    foreach ($variations as $key => $variation) {
                        $variationTitleConcat = "";
                        $selectedOptions = [];
                        foreach ($variation['attributes'] as $key => $value) {
                            $attribute = substr($key, strlen("attribute_"));
                            if (strpos($attribute, "size")) {
                                $attribute = "Size";
                            }
                            $selectedOptions[] = (object)[
                                "name" => ucfirst($attribute),
                                "value" => $value,
                            ];
                            if (!empty($variationTitleConcat)) {
                                $variationTitleConcat .= ' / ';
                            }
                            $variationTitleConcat .= $value;
                        }

                        $variantId = kixxl_wc_generate_resource_id('variant', $variation['variation_id']);
                        $variants_data[] = (object)[
                            'legacyResourceId' => $variantId,
                            'title' => $variationTitleConcat,
                            'price' => empty($variation['display_price']) ? 0.00 : (float)$variation['display_price'],
                            'sku' => $variation['sku'],
                            'displayName' => $product->title . ' - ' . $variation['display_price'] . '/' . $variationTitleConcat,
                            'position' => $key == 0 ? 1 : $key,
                            'selectedOptions' => $selectedOptions,
                        ];
                    }
                } else {
                    $hasOnlyDefaultVariant = true;
                    $variants_data[] = (object)[
                        'legacyResourceId' => kixxl_wc_generate_resource_id('variant', $product->get_id()),
                        'title' => 'Default Title',
                        'price' => empty($product->get_price()) ? 0.00 : (float)$product->get_price(),
                        'sku' => $product->get_sku(),
                        'displayName' => $product->title . ' - Default Title',
                        'position' => 1,
                        'selectedOptions' => (object)[
                            "name" => 'Title',
                            "value" => 'Default Title',
                        ]
                    ];
                }

                $productId = kixxl_wc_generate_resource_id('product', $product->get_id());

                $searched_product = (object)[
                    'id' => $productId,
                    'legacyResourceId' => $productId,
                    'price' => $product->get_price() ?? 0.00,
                    'title' => html_entity_decode($product->get_name(), ENT_QUOTES, 'UTF-8'),
                    'hasOnlyDefaultVariant' => $hasOnlyDefaultVariant,
                    'options' => $selectedOptions ?? [],
                    'priceRangeV2' => (object)[
                        'maxVariantPrice' => (object)[
                            'currencyCode' => get_option('woocommerce_currency'),
                        ],
                        'minVariantPrice' => (object)[
                            'currencyCode' => get_option('woocommerce_currency'),
                        ],
                    ],
                    'featuredImage' => (object)[
                        'url' => $image_url,
                    ],
                    'variants' => (object)[
                        'nodes' => $variants_data,
                    ],
                ];

                $product_data[] = $searched_product;
            }

            $body = (object)[
                "data" => (object)[
                    "products" => (object)[
                        "nodes" => $product_data,
                    ],
                ],
            ];

            $response = (object)[
                "body" => $body,
                "errors" => false,
            ];

            return rest_ensure_response($response);
        } catch (Throwable $exception) {
            return rest_ensure_response((object)[
                'errors' => true,
                'status' => 500,
                'exception' => (object)[
                    'message' => $exception->getMessage(),
                    'code' => $exception->getCode(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString(),
                ]
            ]);
        }
    }

    public function search_product_by_id_using_graphql($request)
    {
        try {
            $search_query = kixxl_wc_decode_resource_id($request['productId']);

            if (empty($search_query)) {
                return rest_ensure_response((object)[
                    'errors' => (object)['message' => __('Search query is required.')],
                    'status' => 400
                ]);
            }

            $product_query_args = [
                'status' => ['publish', 'draft'], // optional, if drafts should be allowed
                'limit' => 1,
                'include' => [(int)$search_query],
            ];
            $products_query = new WC_Product_Query($product_query_args);
            $products = $products_query->get_products();
            $product = $products[0] ?? null;
            if (empty($product)) {
                return rest_ensure_response((object)[
                    "body" => (object)[
                        "data" => (object)[
                            "products" => (object)[
                                "nodes" => (object)[],
                            ],
                        ],
                    ],
                    "errors" => false,
                ]);
            }

            $product_data = [];
            $image_id = $product->get_image_id();
            $image_url = wp_get_attachment_image_url($image_id, 'full');

            $variants_data = [];
            $hasOnlyDefaultVariant = false;
            if ($product->is_type('variable')) {
                $variations = $product->get_available_variations();
                foreach ($variations as $key => $variation) {
                    $variationTitleConcat = "";
                    $selectedOptions = [];
                    foreach ($variation['attributes'] as $key => $value) {
                        $attribute = substr($key, strlen("attribute_"));
                        if (strpos($attribute, "size")) {
                            $attribute = "Size";
                        }
                        $selectedOptions[] = (object)[
                            "name" => ucfirst($attribute),
                            "value" => $value,
                        ];
                        if (!empty($variationTitleConcat)) {
                            $variationTitleConcat .= ' / ';
                        }
                        $variationTitleConcat .= $value;
                    }

                    $variantId = kixxl_wc_generate_resource_id('variant', $variation['variation_id']);

                    $variants_data[] = (object)[
                        'legacyResourceId' => $variantId,
                        'title' => $variationTitleConcat,
                        'price' => empty($variation['display_price']) ? 0.00 : (float)$variation['display_price'],
                        'sku' => $variation['sku'],
                        'displayName' => $product->title . ' - ' . $variation['display_price'] . '/' . $variationTitleConcat,
                        'position' => $key == 0 ? 1 : $key,
                        'selectedOptions' => $selectedOptions,
                    ];
                }
            } else {
                $hasOnlyDefaultVariant = true;
                $variants_data[] = (object)[
                    'legacyResourceId' => kixxl_wc_generate_resource_id('variant', $product->get_id()),
                    'title' => 'Default Title',
                    'price' => empty($product->get_price()) ? 0.00 : (float)$product->get_price(),
                    'sku' => $product->get_sku(),
                    'displayName' => $product->title . ' - Default Title',
                    'position' => 1,
                    'selectedOptions' => (object)[
                        "name" => 'Title',
                        "value" => 'Default Title',
                    ]
                ];
            }

            $productId = kixxl_wc_generate_resource_id('product', $product->get_id());

            $searched_product = (object)[
                'id' => $productId,
                'legacyResourceId' => $productId,
                'price' => $product->get_price() ?? 0.00,
                'title' => $product->get_name(),
                'hasOnlyDefaultVariant' => $hasOnlyDefaultVariant,
                'options' => $selectedOptions,
                'priceRangeV2' => (object)[
                    'maxVariantPrice' => (object)[
                        'currencyCode' => get_option('woocommerce_currency'),
                    ],
                    'minVariantPrice' => (object)[
                        'currencyCode' => get_option('woocommerce_currency'),
                    ],
                ],
                'featuredImage' => (object)[
                    'url' => $image_url,
                ],
                'variants' => (object)[
                    'nodes' => $variants_data,
                ],
            ];

            $product_data[] = $searched_product;

            $body = (object)[
                "data" => (object)[
                    "products" => (object)[
                        "nodes" => $product_data,
                    ],
                ],
            ];

            $response = (object)[
                "body" => $body,
                "errors" => false,
            ];

            return rest_ensure_response($response);
        } catch (Throwable $exception) {
            return rest_ensure_response((object)[
                'errors' => true,
                'status' => 500,
                'exception' => (object)[
                    'message' => $exception->getMessage(),
                    'code' => $exception->getCode(),
                    'file' => $exception->getFile(),
                    'line' => $exception->getLine(),
                    'trace' => $exception->getTraceAsString(),
                ]
            ]);
        }
    }

    public function get_product_meta_fields($request)
    {
        try {
            $product_id = kixxl_wc_decode_resource_id($request['productId']);

            $custom_fields = get_post_meta($product_id) ?? [];

            $metafields = ['nodes' => []];

            $counter = 0;

            foreach ($custom_fields as $key => $custom_field) {
                // @TODO: Get Custom fields properly
                // to exclude pre-defined meta data
                if (substr($key, 0, 1) !== '_') {

                    $metafields['nodes']['node'] = [
                        'namespace' => 'kixxl',
                        'key' => $key,
                        'value' => $custom_field[0],
                    ];

                    $counter++;

                    // Require first 10 metafields
                    if ($counter == 10) {
                        break;
                    }
                }
            }

            return rest_ensure_response([
                "errors" => false,
                "response" => [],
                "status" => 200,
                "body" => [
                    'data' => [
                        'product' => [
                            "metafields" => ['edges' => $metafields]
                        ]
                    ]
                ],
                "link" => null,
            ]);
        } catch (Throwable $exception) {
            return $this->error_response($exception);
        }
    }

    public function get_locations($request)
    {
        try {
            return rest_ensure_response([
                'errors' => false,
                'response' => [],
                'status' => 200,
                'body' => [
                    'data' => [
                        'locations' => [
                            'edges' => [
                                [
                                    'node' => [
                                        'id' => "gid://woocommerce/Location/" . kixxl_wc_generate_unique_id()
                                    ]
                                ]
                            ]
                        ]
                    ]
                ]
            ]);
        } catch (Throwable $exception) {
            return $this->error_response($exception);
        }
    }

    public function get_publications($request)
    {
        try {
            return rest_ensure_response([
                'errors' => false,
                'response' => [],
                'status' => 200,
                'body' => [
                    'data' => [
                        'publications' => [
                            'nodes' => [
                                'id' => "gid://woocommerce/Publication/" . kixxl_wc_generate_unique_id(),
                                'name' => "Online Store",
                                'catalog' => null,
                            ]
                        ]
                    ]
                ]
            ]);
        } catch (Throwable $exception) {
            return $this->error_response($exception);
        }
    }

    public function send_invoice($request)
    {
        try {
            return rest_ensure_response([
                'errors' => false,
                'response' => [],
                'status' => 200,
                'body' => []
            ]);
        } catch (Throwable $exception) {
            return $this->error_response($exception);
        }
    }
}
