<?php

/*
Plugin Name: Kixxl WooCommerce
Plugin URI: https://kixxl.com/
Description: The AI-Powered Gang Sheet Builder
Version: 13.16.2
Author: Kixxl
Author URI: https://kixxl.com/
Requires Plugins: woocommerce
*/

register_activation_hook(__FILE__, 'gangify_flush_rewrite_rules');
function gangify_flush_rewrite_rules()
{
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(__('Kixxl requires WooCommerce to be installed and active.', 'Kixxl'));
    }

    add_rewrite_rule('^apps/gangify/(.+)/?', 'index.php?gangify_endpoint=1&gangify_path=$matches[1]', 'top');
    flush_rewrite_rules();
}

register_deactivation_hook(__FILE__, 'flush_rewrite_rules');
add_action('init', 'gangify_add_rewrite_rules');
function gangify_add_rewrite_rules()
{
    add_rewrite_rule('^apps/gangify/(.+)/?', 'index.php?gangify_endpoint=1&gangify_path=$matches[1]', 'top');
}

// Register query vars
add_filter('query_vars', 'gangify_register_query_vars');
function gangify_register_query_vars($vars)
{
    $vars[] = 'gangify_endpoint';
    $vars[] = 'gangify_path';
    return $vars;
}

add_action('template_redirect', 'gangify_handle_endpoint');
function gangify_handle_endpoint()
{
    if (get_query_var('gangify_endpoint') == 1) {
        $path = get_query_var('gangify_path');
        $query_params = $_GET;
        unset($query_params['gangify_endpoint']);
        unset($query_params['gangify_path']);
        gangify_process_request($path, $query_params);
        exit;
    }
}

function gangify_process_request($path, $query_params = [])
{
    ini_set('display_errors', 'Off');
    kixxl_wc_init_proxy('https://app.gangifyapp.com/proxy');
    $query_params['store'] = preg_replace('#^https?://#', '', get_site_url());
    $query_params['platform'] = 'woocommerce';
	
	$current_user_id = get_current_user_id();
	$resource_id = kixxl_wc_generate_resource_id('customer', $current_user_id);
	$guest_resource_id = kixxl_wc_generate_resource_id('customer', 0);
	
	$query_params['logged_in_customer_id'] =
    	($resource_id == $guest_resource_id || !empty($query_params['staff'])) ? null : $resource_id;
    kixxl_wc_proxy_request($path, $query_params);
}


// Require necessary classes and functions safely
$includes = [
    'includes/class-product-details-controller.php',
    'includes/class-order-details-controller.php',
    'includes/class-general-routes-controller.php',
    'includes/class-gang-sheet-button.php',
    'functions.php',
    'config.php'
];

foreach ($includes as $file) {
    $path = plugin_dir_path(__FILE__) . $file;
    if (file_exists($path)) {
        require_once $path;
    } else {
        error_log("Missing required file: $file");
    }
}

/**
 * Get configuration settings for Kixxl WooCommerce plugin.
 *
 * @return array Plugin configuration data.
 */
try {
    // Include the configuration file for plugin settings
    $config = kixxl_wc_get_config();
    try {
        $shop = kixxl_wc_get_shop($config);
    } catch (Exception $e) {
        error_log('Kixxl WC Shop Error: ' . $e->getMessage());
    }
} catch (Exception $e) {
    error_log('Error: Unable to retrieve configuration.');
}

// Function that registers the user to the Kixxl app when the plugin is activated.
register_activation_hook(__FILE__, 'kixxl_wc_register_user_to_kixxl_app');

// Function that handles cleanup operations when the plugin is uninstalled.
register_uninstall_hook(__FILE__, 'kixxl_wc_plugin_uninstall');

/**
 * Start a session during the WordPress initialization.
 * This ensures that session-related operations are supported in the plugin.
 */
add_action( 'init', 'kixxl_start_session', 1 );

// Add custom admin menu
add_action('admin_menu', 'kixxl_wc_kixxl_add_admin_menu');


/**
 * Initializes the Routes Controller.
 * This controller handles the registration of routes for general, order and product-related actions.
 *
 * @return void
 */

// Require necessary authentication functions
function kixxl_wc_verify_api_request(WP_REST_Request $request)
{
    $request_shop_origin = $request->get_param('store') ?? $request->get_param('shop');
    // Get store name from WordPress options
    $stored_shop_origin = get_option('_kixxl_' . $request_shop_origin . '_shop_id'); // Replace with actual option name
    if (empty($stored_shop_origin)) {
        error_log('Invalid store: ' . $request_shop_origin);
        wp_send_json_error(['message' => 'Unauthorized store'], 401);
        exit;
    }
}

// Protect API routes
add_action('rest_api_init', function () {
    $controllers = [
        new Product_Details_Controller(),
        new Order_Details_Controller(),
        new General_Routes_Controller()
    ];

    foreach ($controllers as $controller) {
        add_action('rest_pre_dispatch', function ($result, $server, $request) use ($controller) {
            $route = $request->get_route();
            if (strpos($route, '/kwc/') !== false) {
                kixxl_wc_verify_api_request($request);
            }
            return $result;
        }, 10, 3);

        $controller->register_routes();
    }
});


/**
 * This hook triggers when an order status changes to 'processing'.
 * It allows the plugin to handle order creation and synchronization with Kixxl.
 *
 * @param int $order_id The ID of the WooCommerce order being processed.
 *
 * @return void
 */
add_action('woocommerce_new_order', 'kixxl_wc_order_create', 10, 2);


add_action('update_option_WPLANG', 'on_site_language_changed', 10, 2);

/**
 * This hook triggers when an order status changes.
 * It allows the plugin to perform actions based on status updates.
 *
 * @param int $order_id The ID of the WooCommerce order.
 * @param string $old_status The old status of the order.
 * @param string $new_status The new status of the order.
 *
 * @return void
 */
add_action('woocommerce_order_status_changed', 'kixxl_wc_order_status_changed', 10, 3);

/**50
 * This hook triggers when a product is updated in WooCommerce.
 * It allows the plugin to handle product updates and synchronization with Kixxl.
 *
 * @param int $product_id The ID of the WooCommerce product being updated.
 *
 * @return void
 */
add_action('woocommerce_update_product', 'kixxl_wc_product_updated', 10, 1);

/**
 * Get KIXXL_WC Config
 *
 * Fetches and caches the plugin's configuration data from the config file.
 *
 * @return array The configuration data.
 */
function kixxl_wc_get_config()
{
    static $config = null;
//     $cache_key = 'kixxl_wc_config';
//     $config = get_transient($cache_key);

    if ($config === false || $config === null) {
        $path = plugin_dir_path(__FILE__) . 'config.php';
        if (file_exists($path)) {
            $config = include $path;
            if (!is_array($config)) {
                error_log('Invalid config file format.');
                $config = [];
            }
//             set_transient($cache_key, $config, 3600); // Cache for 1 hour
        } else {
            error_log('Config file missing.');
        }
    }
    return $config;
}

/**
 * Enqueue plugin-specific JavaScript.
 */
function kixxl_wc_kixxl_woocommerce_enqueue_scripts()
{
    // Register the script
    wp_register_script(
        'kixxl-woocommerce-scripts', // Handle
        plugin_dir_url(__FILE__) . 'js/gang-sheet-button.js',
        // Path to the script
        array('jquery'), // Dependencies (optional)
        '1.0.0', // Version (optional)
        true // Load in footer (optional)
    );

    // Enqueue the script
    wp_enqueue_script('kixxl-woocommerce-scripts');

}

add_action('wp_enqueue_scripts', 'kixxl_wc_kixxl_woocommerce_enqueue_scripts');

/**
 * Localize PHP variables to pass data to the front-end script.
 */
function localize_script_variables()
{
    global $config, $shop, $wpdb;
    /*if (!current_user_can('read')) {
        return;
    }*/
    $customer_data = null;
    $customer_id = null;

    if (is_user_logged_in()) {
        $user = wp_get_current_user();
        $user_email = $user->user_email;

        $table_name = $wpdb->prefix . 'wc_customer_lookup';
        $customer_data = array();

        if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") == $table_name) {
            $customer = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM $table_name WHERE email = %s",
                $user_email
            ));

            if ($customer) {
                $customer_data = array(
                    'id' => $customer->customer_id,
                    'email' => $customer->email,
                    'first_name' => $customer->first_name,
                    'last_name' => $customer->last_name,
                );
            }
        }

        $customer_id = kixxl_wc_generate_resource_id('customer', get_current_user_id());
        if (empty($customer_id) && !empty($customer_data)) {
            $customer_id = $customer_data['id'] ?? null;
        }
    }
	
	$current_user = wp_get_current_user();
    // Provide fallback/defaults so `php_vars` always exists
    wp_localize_script('kixxl-woocommerce-scripts', 'php_vars', array(
        'config' => $config ?? [],
        'shop' => $shop ?? '',
        'customer' => $customer_data ?? null,
        'customer_id' => $customer_id ?? null,
		'current_user' => array(
			'ID' => $current_user->ID,
			'user_login' => $current_user->user_login,
			'user_email' => $current_user->user_email,
		),
    ));
}

add_action('wp_enqueue_scripts', 'localize_script_variables');


/**
 * Parse the product item from the query string.
 */
function parse_product_item($item)
{
    $parsed_data = [];
    $item = str_replace("?rest_route=", "?rest_route[]", $item);
    $item = str_replace("&edit-sheet=", "&edit-sheet[]", $item);
    $item = str_replace("&store=", "&store[]", $item);
    $item = str_replace("&platform=", "&platform[]", $item);
    $item = str_replace("?sheet_key=", "?sheet_key[]", $item);
    $item = str_replace("&json=", "&json[]", $item);
	$item = str_replace("&variant=", "&variant[]", $item);
    $item = str_replace("&product=", "&product[]", $item);
    $item = str_replace("&is_admin=", "&is_admin[]", $item);
    $parts = explode('|', $item);

    foreach ($parts as $part) {
        $key_value = explode('=', $part);
		
        if (count($key_value) === 2) {
            $item = $key_value[1];
            $item = str_replace("?rest_route[]", "?rest_route=", $item);
            $item = str_replace("&edit-sheet[]", "&edit-sheet=", $item);
            $item = str_replace("&store[]", "&store=", $item);
            $item = str_replace("&platform[]", "&platform=", $item);
            $item = str_replace("?sheet_key[]", "?sheet_key=", $item);	
            $item = str_replace("&json[]", "&json=", $item);
            $item = str_replace("&variant[]", "&variant=", $item);
            $item = str_replace("&product[]", "&product=", $item);
            $item = str_replace("&is_admin[]", "&is_admin=", $item);

            $parsed_data[sanitize_text_field($key_value[0])] = sanitize_text_field($item);
        }
    }
	
    return $parsed_data;
}

/**
 * Sanitize tags for safe storage.
 */
function sanitize_tags($tags)
{
    if (strpos($tags, 'used-balance;') !== false) {
        return str_replace(';', '=', $tags);
    }
    return $tags;
}

/**
 * Prepare cart item data with custom attributes.
 */
function prepare_cart_item_data($parsed_data)
{
    $custom_data = [
		'_app_name' => sanitize_text_field($parsed_data['_app_name'] ?? ''),
		'_description' => sanitize_text_field($parsed_data['_description'] ?? ''),
		'sheet_preview' => esc_url($parsed_data['sheet_preview'] ?? ''),
		'_actual_gang_sheet' => esc_url($parsed_data['_actual_gang_sheet'] ?? ''),
		'_has_overlapping_images' => sanitize_text_field($parsed_data['_has_overlapping_images'] ?? ''),
		'_has_low_dpi_images' => sanitize_text_field($parsed_data['_has_low_dpi_images'] ?? ''),
		'_edit_as_admin' => esc_url($parsed_data['_edit_as_admin'] ?? ''),
		'sheet_type' => sanitize_text_field($parsed_data['sheet_type'] ?? 'created'),
		'variant_id' => intval($parsed_data['variantId'] ?? 0),
		'price' => floatval($parsed_data['price'] ?? 0),
		'tags' => sanitize_tags($parsed_data['tags'] ?? ''),
		'quantity' => max(1, intval($parsed_data['quantity'] ?? 0)), // Ensuring min quantity is 1
		'unique_key' => md5(serialize($parsed_data) . microtime() . rand()),
	];

    if ( ! empty( $parsed_data['edit_sheet'] ) ) {
        $custom_data['edit_sheet'] = esc_url( $parsed_data['edit_sheet'] );
    }

    // Only add sheet_size if it exists and is not empty
    if(!empty($parsed_data['sheet_size'])) {
        $custom_data['sheet_size'] = sanitize_text_field($parsed_data['sheet_size']);
    }

    //  Dynamically add all keys starting with 'extra_' (YayExtra)
    foreach ( $parsed_data as $key => $value ) {
        if ( strpos( $key, 'extra_' ) === 0 ) {
            if ( is_string( $value ) ) {
                // Replace + with, and sanitize
                $value = str_replace( '+', ',', $value );
                $custom_data[ $key ] = sanitize_text_field( $value );
            } else {
                $custom_data[ $key ] = $value;
            }
        }
    }

    return ['custom_data' => $custom_data];
}

/**
 * Return content (custom attributes) stored in a file in includes folder.
 */
function kixxl_create_url($filename)
{
    $file = plugin_dir_path(__FILE__) . 'includes/' . $filename;
    if (!file_exists($file)) {
        return '';
    }
    return file_get_contents($file);
}

/**
 * Add multiple products to the WooCommerce cart.
 *
 * This function adds multiple products to the cart based on the query string in the URL.
 * The products are identified by 'variantId', and additional custom data is passed with the cart item.
 */
function kixxl_add_multiple_products_to_cart()
{
    global $woocommerce;

    if (isset($_GET['add-to-cart-multiple'])) {

        $file = $_GET['file'];
        $tags = $_GET['tags'] ?? '';

        $products = kixxl_create_url($file);

        // Parse products string
        $product_items = explode(',', $products);
        foreach ($product_items as $item) {
			
            $parsed_data = parse_product_item($item);
            $parsed_data['tags'] = $tags;
            // Accessing the parsed values
            $product_id = isset($parsed_data['variantId']) ? (int)$parsed_data['variantId'] : 0;
            $quantity = isset($parsed_data['quantity']) ? (int)$parsed_data['quantity'] : 0;

            if ($product_id > 0 && $quantity > 0) {
                // Add product to cart with custom data
                $cart_item_data = prepare_cart_item_data($parsed_data);
                WC()->cart->add_to_cart($product_id, $quantity, 0, array(), $cart_item_data);
            }
        }
        // redirect to cart page
        wp_safe_redirect(wc_get_cart_url());
        exit;
    }
}

add_action('wp_loaded', 'kixxl_add_multiple_products_to_cart');

add_filter('woocommerce_order_item_display_meta_value', 'kixxl_override_link_meta_display', 10, 3);

function kixxl_override_link_meta_display($display_value, $meta, $item) {
    // Keys you want to replace with links
    $link_keys = ['sheet_preview', '_actual_gang_sheet', 'edit_sheet', '_edit_as_admin'];

    if (in_array($meta->key, $link_keys) && filter_var($meta->value, FILTER_VALIDATE_URL)) {
        return '<a href="' . esc_url($meta->value) . '" target="_blank" rel="noopener noreferrer">Link</a>';
    }
    return $display_value;
}

/**
 * Add custom metadata to cart item.
 *
 * This function processes and adds custom data like `_app_name` and `_description` to each cart item.
 *
 * @param array $cart_item_data The existing cart item data.
 * @param int $product_id The product ID.
 * @param int $variation_id The variation ID.
 *
 * @return array Modified cart item data with additional custom data.
 */
function kixxl_add_cart_item_data(array $cart_item_data, int $product_id, int $variation_id): array
{
    if (isset($cart_item_data['_app_name'])) {
        $cart_item_data['_app_name'] = sanitize_text_field($cart_item_data['_app_name']);
    }
    if (isset($cart_item_data['_description'])) {
        $cart_item_data['_description'] = sanitize_text_field($cart_item_data['_description']);
    }
    if (isset($cart_item_data['sheet_preview'])) {
        $cart_item_data['sheet_preview'] = esc_url_raw($cart_item_data['sheet_preview']);
    }
    if (isset($cart_item_data['_actual_gang_sheet'])) {
        $cart_item_data['_actual_gang_sheet'] = esc_url_raw($cart_item_data['_actual_gang_sheet']);
    }
    if (isset($cart_item_data['edit_sheet'])) {
        $cart_item_data['edit_sheet'] = sanitize_text_field($cart_item_data['edit_sheet']);
    }
    if (isset($cart_item_data['sheet_type'])) {
        $cart_item_data['sheet_type'] = sanitize_text_field($cart_item_data['sheet_type']);
    }

    // Dynamically handle any keys starting with 'extra_'
    foreach ( $cart_item_data as $key => $value ) {
        if ( strpos( $key, 'extra_' ) === 0 ) {
            $new_key = substr( $key, 6 ); // remove 'extra_'

            if ( is_string( $value ) ) {
                // Replace + with, and sanitize
                $value = str_replace( '+', ',', $value );
                $cart_item_data[ $new_key ] = sanitize_text_field( $value );
            } else {
                $cart_item_data[ $new_key ] = $value;
            }

            // remove the original prefixed key
            unset( $cart_item_data[ $key ] );
        }
    }


    return $cart_item_data;
}

/**
 * Save custom product data to WooCommerce cart.
 *
 * This function handles adding multiple products with custom attributes like 'sheet_preview', '_app_name', and '_description' to the cart.
 *
 * @param array $cart_item_data Cart item data.
 * @param int $product_id Product ID.
 *
 * @return array Modified cart item data.
 */
function kixxl_wc_save_custom_product_data(array $cart_item_data, int $product_id): array
{
    WC()->cart->get_cart();
    if (!isset($cart_item_data['custom_data'])) {
        return $cart_item_data;
    }

    $cart_item_data['custom_data']['unique_key'] = $cart_item_data['custom_data']['unique_key'] ?? md5(microtime() . rand());
    WC()->session->set('custom_variations', $cart_item_data['custom_data']);

    return $cart_item_data;
}

add_filter('woocommerce_add_cart_item_data', 'kixxl_add_cart_item_data', 10, 3);
add_filter('woocommerce_add_cart_item_data', 'kixxl_wc_save_custom_product_data', 10, 2);

/**
 * Set custom price for the cart items.
 *
 * This function sets a custom price for each product in the cart, based on the price passed in the cart item data.
 *
 * @param WC_Cart $cart The WooCommerce cart object.
 */
function set_custom_price(WC_Cart $cart)
{
    foreach ($cart->get_cart() as $cart_item) {
        if (isset($cart_item['custom_data']['price'])) {
			$cart_item['data']->set_price($cart_item['custom_data']['price']);
		}
    }
}

add_action('woocommerce_before_calculate_totals', 'set_custom_price', 20, 1);


/**
 * Adds custom attributes to the cart and checkout items.
 * These attributes are shown to the user during the checkout process.
 *
 * @param array $cart_data Array of existing cart data.
 * @param array $cart_item The WooCommerce cart item data.
 *
 * @return array Updated cart data with custom attributes.
 */
function kixxl_wc_customizing_cart_item_data( array $cart_data, array $cart_item ): array {
    $custom_data = $cart_item['custom_data'] ?? [];

    //  Static keys you always want to show
    $static_keys = [ '_app_name', 'sheet_preview', 'edit_sheet', 'sheet_type' ];

    foreach ( $custom_data as $key => $value ) {
        // Handle static keys
        if ( in_array( $key, $static_keys, true ) ) {
            if ( in_array( $key, [ 'sheet_preview', 'edit_sheet' ], true ) ) {
                $value = wp_kses_post(sprintf(
                        '<a href="%s" target="_blank">%s</a>',
                        esc_url($value),
                        esc_html($value))
                );
            }

            $cart_data[] = [
                'name' => $key,
                'value' => $value,
            ];
        }

        // Handle dynamic "extra_" prefixed keys
        if ( strpos( $key, 'extra_' ) === 0 ) {
            $new_key = substr( $key, 6 ); // remove "extra_"

            if ( is_string( $value ) ) {
                // Replace + with, before escaping
                $value = str_replace( '+', ',', $value );
                $cart_data[] = [
                    'name'  => esc_html( $new_key ),
                    'value' => esc_html( $value ),
                ];
            } else {
                $cart_data[] = [
                    'name'  => esc_html( $new_key ),
                    'value' => wp_json_encode( $value ),
                ];
            }
    }

    }

    return $cart_data;
}

add_filter('woocommerce_get_item_data', 'kixxl_wc_customizing_cart_item_data', 10, 2);

/**
 * Save custom data from the cart to WooCommerce order items.
 * Ensures that attributes like 'sheet_preview', '_app_name', '_description' are saved in order metadata.
 *
 * @param WC_Order_Item $item The WooCommerce order item object.
 * @param string $values The WooCommerce cart item values.
 *
 * @return void
 */
function kixxl_wc_save_custom_order_item_meta_data(WC_Order_Item $item, string $values)
{
    if ( isset( $item->legacy_values['custom_data'] ) && is_array( $item->legacy_values['custom_data'] ) ) {
        foreach ($item->legacy_values['custom_data'] as $key => $value) {

            // Handle special case for 'tags'
            if ( $key === 'tags' && strpos( $value, 'used-balance;' ) !== false ) {
                    $value = str_replace(';', '=', $value);
                }

            // Clean the key if it starts with "extra_"
            if ( strpos( $key, 'extra_' ) === 0 ) {
                $key = substr( $key, 6 ); // remove 'extra_'
                $value = str_replace( '+', ',', $value );
            }

            // Save the cleaned meta
            $item->update_meta_data($key, $value);
        }

        // Save item meta once after loop (no need to save inside loop)
            $item->save();
        }
    }

add_action('woocommerce_checkout_create_order_line_item', 'kixxl_wc_save_custom_order_item_meta_data', 10, 2);

/**
 * This function fetches the plugin update information from a remote server
 *
 * @return mixed
 */
function get_remote_plugin_update_info()
{
    global $config;

    // If config is missing or doesn't have PLUGIN_UPDATE_URL, use the default URL
    $plugin_update_url = !empty($config['PLUGIN_UPDATE_URL'])
        ? $config['PLUGIN_UPDATE_URL']
        : "https://kixxl.com/woocommerce-plugin/kixxl.json";

    // Append param as a cache buster
    $url = add_query_arg('x', time(), $plugin_update_url);

    $response = wp_remote_get($url, [
		'sslverify'   => false,
		'timeout'     => 20,
		'redirection' => 5,
		'headers'     => [
			'Accept-Encoding' => 'identity',
			'User-Agent'      => 'Mozilla/5.0 (compatible; MyBot/1.0)',
			'Cache-Control'   => 'no-cache',
			'Pragma'          => 'no-cache',
		]
    ]);

    if (is_wp_error($response) || wp_remote_retrieve_response_code($response) != 200) {
        $error_message = $response->get_error_message();
        error_log('Failed to fetch plugin update information: ' . $error_message);
        return null;
    }

    return json_decode(wp_remote_retrieve_body($response));
}

/**
 * This function checks for plugin updates by comparing the version number
 *
 * @param $checked_data
 *
 * @return mixed
 */
function check_for_plugin_update($checked_data)
{
    if (empty($checked_data->checked)) {
        return $checked_data;
    }

    $plugin_slug = plugin_basename(__FILE__);
    $update_info = get_remote_plugin_update_info();

    if (isset($checked_data->checked[$plugin_slug]) && !empty($update_info)
        && version_compare($update_info->version, $checked_data->checked[$plugin_slug], '>')) {
        $checked_data->response[$plugin_slug] = (object)array(
            'slug' => $plugin_slug,
            'new_version' => $update_info->version,
            'url' => $update_info->download_url,
            'package' => $update_info->download_url,
        );
    }

    return $checked_data;
}

add_filter('site_transient_update_plugins', 'check_for_plugin_update');

/**
 * This function fetches the plugin update information from a remote server
 *
 * @param $res
 * @param $action
 * @param $args
 *
 * @return mixed
 */
function plugin_api_call($res, $action, $args)
{
    if ($action !== 'plugin_information' || $args->slug !== plugin_basename(__FILE__)) {
        return $res;
    }

    $plugin_info = get_remote_plugin_update_info();

    return (object)[
        'name' => 'Kixxl WooCommerce',
        'slug' => plugin_basename(__FILE__),
        'version' => $plugin_info->version,
        'sections' => [
            'description' => $plugin_info->sections->description,
        ],
    ];
}

add_filter('plugins_api', 'plugin_api_call', 10, 3);
add_filter('auto_update_plugin', '__return_true');